/* Copyright (c) 1992 Henry Spencer.
 * Copyright (c) 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Henry Spencer of the University of Toronto.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)regex.h	8.1 (Berkeley) 6/2/93
 */

#ifndef _REGEX_H_
#define	_REGEX_H_

#if defined (_WIN32)
#  include <BaseTsd.h>
#endif

/* Pre VS2015, MSVC doesn't provide snprintf */
#if defined(_MSC_VER) && _MSC_VER < 1900
#  define snprintf _snprintf
#endif

#include <limits.h>
#include <stddef.h>

#if !defined(ssize_t)
   typedef ptrdiff_t ssize_t;
#  define HAVE_SSIZE_T 1
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

/* We want 64 bit (large file) I/O capabilities whenever they are available.
 * Always define this before we include sys/types.h */
#ifndef _FILE_OFFSET_BITS
#  define _FILE_OFFSET_BITS 64
#endif
#include <sys/types.h>

/* off_t is 32 bit size even on 64 bit Windows. In the past we have tried to
 * force off_t to be 64 bit but this is failing on newer Windows/Visual Studio
 * verions in 2020 - therefore, we instead introduce the b_off_t define to
 * properly substitute the correct numerical type for the correct platform.  */
#if defined(_WIN64)
#  include <sys/stat.h>
#  define regoff_t __int64
#  define fseek _fseeki64
#  define ftell _ftelli64
#  define fstat _fstati64
#  define stat  _stati64
#elif defined (_WIN32)
#  include <sys/stat.h>
#  define regoff_t _off_t
#  define fstat _fstat
#  define stat  _stat
#else
#  define regoff_t off_t
#endif


#ifndef __BEGIN_DECLS
# ifdef __cplusplus
#   define __BEGIN_DECLS   extern "C" {
#   define __END_DECLS     }
# else
#   ifndef __BEGIN_DECLS
#     define __BEGIN_DECLS
#   endif
#   ifndef __END_DECLS
#     define __END_DECLS
#   endif
# endif
#endif


#ifndef REGEX_EXPORT
#  if defined(_WIN32) && !defined(__CYGWIN__) && defined(BRLCAD_DLL)
#    ifdef REGEX_EXPORT_DLL
#      define REGEX_EXPORT __declspec(dllexport)
#    else
#      define REGEX_EXPORT __declspec(dllimport)
#    endif
#  else
#    define REGEX_EXPORT
#  endif
#endif

/* types */



typedef struct {
	int re_magic;
	size_t re_nsub;		/* number of parenthesized subexpressions */
	const char *re_endp;	/* end pointer for REG_PEND */
	struct re_guts *re_g;	/* none of your business :-) */
} regex_t;

typedef struct {
	regoff_t rm_so;		/* start of match */
	regoff_t rm_eo;		/* end of match */
} regmatch_t;

/* regcomp() flags */
#define	REG_BASIC	0000
#define	REG_EXTENDED	0001
#define	REG_ICASE	0002
#define	REG_NOSUB	0004
#define	REG_NEWLINE	0010
#define	REG_NOSPEC	0020
#define	REG_PEND	0040
#define	REG_DUMP	0200

/* regerror() flags */
#define	REG_NOMATCH	 1
#define	REG_BADPAT	 2
#define	REG_ECOLLATE	 3
#define	REG_ECTYPE	 4
#define	REG_EESCAPE	 5
#define	REG_ESUBREG	 6
#define	REG_EBRACK	 7
#define	REG_EPAREN	 8
#define	REG_EBRACE	 9
#define	REG_BADBR	10
#define	REG_ERANGE	11
#define	REG_ESPACE	12
#define	REG_BADRPT	13
#define	REG_EMPTY	14
#define	REG_ASSERT	15
#define	REG_INVARG	16
#define	REG_ATOI	255	/* convert name to number (!) */
#define	REG_ITOA	0400	/* convert number to name (!) */

/* regexec() flags */
#define	REG_NOTBOL	00001
#define	REG_NOTEOL	00002
#define	REG_STARTEND	00004
#define	REG_TRACE	00400	/* tracing of execution */
#define	REG_LARGE	01000	/* force large representation */
#define	REG_BACKR	02000	/* force use of backref code */

#define REGEX_PREFIX 1
#define REGEX_PREFIX_STR brl_
#ifdef REGEX_PREFIX
/* Allow a user configurable prefix string, defaulting to "z_" */
#  if !defined(REGEX_PREFIX_STR)
#    define REGEX_PREFIX_STR regex_
#  endif
#  define REGEXLIB_CONCAT2(a, b) a ## b
#  define REGEXLIB_CONCAT(a, b) REGEXLIB_CONCAT2(a,b)
#  define REGEX_ADD_PREFIX(b) REGEXLIB_CONCAT(REGEX_PREFIX_STR,b)

#  define regcomp  REGEX_ADD_PREFIX(regcomp)
#  define regerror REGEX_ADD_PREFIX(regerror)
#  define regexec  REGEX_ADD_PREFIX(regexec)
#  define regfree  REGEX_ADD_PREFIX(regfree)
#endif

__BEGIN_DECLS
REGEX_EXPORT int	regcomp (regex_t *, const char *, int);
REGEX_EXPORT size_t	regerror (int, const regex_t *, char *, size_t);
REGEX_EXPORT int	regexec (const regex_t *, const char *, size_t, regmatch_t [], int);
REGEX_EXPORT void	regfree (regex_t *);
__END_DECLS

#endif /* !_REGEX_H_ */

/*
 * Local Variables:
 * mode: C
 * tab-width: 8
 * indent-tabs-mode: t
 * c-file-style: "stroustrup"
 * End:
 * ex: shiftwidth=4 tabstop=8
 */
