/*
 * PROJECT:     ReactOS Setup Library
 * LICENSE:     GPL-2.0+ (https://spdx.org/licenses/GPL-2.0+)
 * PURPOSE:     MBR and GPT Partition types
 * COPYRIGHT:   Copyright 2018-2020 Hermes Belusca-Maito
 */

#include "precomp.h"
#include "partinfo.h"

/* MBR PARTITION TYPES ******************************************************/

/*
 * This partition type list is based from:
 * - the kernelDisk.c module of the Visopsys Operating System (see license below),
 * - Paragon Hard-Disk Manager,
 * - Haiku OS (Copyright 2003-2011, Haiku, Inc., under the terms of the MIT License)
 *   https://git.haiku-os.org/haiku/tree/src/add-ons/kernel/partitioning_systems/intel/PartitionMap.cpp#n52
 * - and the following websites:
 *   http://www.win.tue.nl/~aeb/partitions/partition_types-1.html
 *   https://en.wikipedia.org/wiki/Partition_type#List_of_partition_IDs
 *   https://www.magnumdb.com/search?q=PARTITION_*
 */
/*
 * kernelDisk.c
 *
 * Visopsys Operating System
 * Copyright (C) 1998-2020 J. Andrew McLaughlin
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/* Known MBR partition type codes and descriptions */
const MBR_PARTITION_TYPE MbrPartitionTypes[NUM_MBR_PARTITION_TYPES] =
{
    { 0x00, "(Empty)" },                                    // PARTITION_ENTRY_UNUSED
    { 0x01, "FAT12" },                                      // PARTITION_FAT_12
    { 0x02, "XENIX root" },                                 // PARTITION_XENIX_1 - "XENIX Type 1"
    { 0x03, "XENIX /usr" },                                 // PARTITION_XENIX_2 - "XENIX Type 2"
    { 0x04, "FAT16 (< 32 MB)" },                            // PARTITION_FAT_16
    { 0x05, "Extended" },                                   // PARTITION_EXTENDED - "Extended partition MS-DOS V4"
    { 0x06, "FAT16" },                                      // PARTITION_HUGE     - "Huge partition MS-DOS V4"
    { 0x07, "NTFS/HPFS/exFAT" },                            // PARTITION_IFS
    { 0x08, "OS/2 or AIX boot partition" },
    { 0x09, "AIX data partition" },
    { 0x0A, "OS/2 Boot Manager" },                          // PARTITION_OS2BOOTMGR - "OS/2 Boot Manager/OPUS/Coherent swap"
    { 0x0B, "FAT32" },                                      // PARTITION_FAT32
    { 0x0C, "FAT32 (LBA)" },                                // PARTITION_FAT32_XINT13
    { 0x0E, "FAT16 (LBA)" },                                // PARTITION_XINT13
    { 0x0F, "Extended (LBA)" },                             // PARTITION_XINT13_EXTENDED
    { 0x10, "OPUS" },
    { 0x11, "Hidden FAT12" },
    { 0x12, "FAT diagnostic (Compaq)" },                    // PARTITION_EISA - "EISA partition"
    { 0x13, "BTRON" },
    { 0x14, "Hidden FAT16 (< 32 MB)" },
    { 0x16, "Hidden FAT16" },
    { 0x17, "Hidden NTFS/HPFS" },
    { 0x18, "AST SmartSleep" },
    { 0x1B, "Hidden FAT32" },
    { 0x1C, "Hidden FAT32 (LBA)" },
    { 0x1E, "Hidden FAT16 (LBA)" },
    { 0x24, "NEC DOS 3.x" },
    { 0x27, "Microsoft Recovery partition" },               // PARTITION_MSFT_RECOVERY  - "Microsoft recovery partition"
    { 0x28, "Main OS partition" },                          // PARTITION_MAIN_OS        - "Main OS partition"
    { 0x29, "OS Data partition" },                          // PARTIITON_OS_DATA        - "OS data partition"
    { 0x2A, "AtheOS File System (AFS)" },                   // PARTITION_PRE_INSTALLED  - "PreInstalled partition"
    { 0x2B, "SyllableSecure (SylStor)" },                   // PARTITION_BSP            - "BSP partition"
    { 0x2C, "DPP partition" },                              // PARTITION_DPP            - "DPP partition"
    { 0x2D, "Windows System partition" },                   // PARTITION_WINDOWS_SYSTEM - "Windows system partition"
    { 0x32, "NOS" },
    { 0x35, "JFS on OS/2 or eCS" },
    { 0x38, "THEOS v3.2 2GB partition" },
    { 0x39, "Plan 9" },
    { 0x3A, "THEOS v4 4GB partition" },
    { 0x3B, "THEOS v4 extended partition" },
    { 0x3C, "PartitionMagic recovery partition" },
    { 0x3D, "Hidden NetWare" },
    { 0x40, "Venix 80286 or Lynx" },
    { 0x41, "PowerPC PReP boot" },                          // PARTITION_PREP - "PowerPC Reference Platform (PReP) Boot Partition"
    { 0x42, "Win2K Dynamic Volume extended" },              // PARTITION_LDM  - "Logical Disk Manager partition"
    { 0x43, "Old Linux" },
    { 0x44, "GoBack" },
    { 0x45, "Priam or Boot-US Boot Manager" },
    { 0x4D, "QNX4.x" },
    { 0x4E, "QNX4.x 2nd partition" },
    { 0x4F, "QNX4.x 3rd partition" },
    { 0x50, "OnTrack Disk Manager R/O" },
    { 0x51, "OnTrack Disk Manager R/W or Novell" },
    { 0x52, "CP/M" },
    { 0x53, "OnTrack DM6 Aux3" },
    { 0x54, "OnTrack DM6 Dynamic Drive Overlay" },          // PARTITION_DM      - "OnTrack Disk Manager partition"
    { 0x55, "EZ-Drive" },                                   // PARTITION_EZDRIVE - "EZ-Drive partition"
    { 0x56, "Golden Bow VFeature Partitioned Volume" },
    { 0x5C, "Priam EDisk" },
    { 0x61, "SpeedStor" },
    { 0x62, "Pick" },
    { 0x63, "GNU HURD or Unix System V (SCO, ISC Unix, UnixWare)" }, // PARTITION_UNIX
    { 0x64, "Novell NetWare 286, 2.xx" },
    { 0x65, "Novell NetWare 386, 3.xx or 4.xx" },
    { 0x66, "Novell NetWare SMS partition" },
    { 0x67, "Novell" },
    { 0x68, "Novell" },
    { 0x69, "Novell NetWare 5+" },
    { 0x70, "DiskSecure Multi-Boot" },
    { 0x75, "IBM PC/IX" },
    { 0x78, "XOSL boot loader" },
    { 0x7E, "Veritas VxVM public" },
    { 0x7F, "Veritas VxVM private" },
    { 0x80, "Old MINIX" },
    { 0x81, "Linux or MINIX" },
    { 0x82, "Linux swap or Solaris" },
    { 0x83, "Linux Native" },
    { 0x84, "Hibernate" },                                  // PARTITION_HIBERNATION - "Hibernation partition for laptops"
    { 0x85, "Linux Extended" },
    { 0x86, "FAT16 mirrored" },
    { 0x87, "NTFS/HPFS mirrored" },
    { 0x88, "Linux plaintext partition table" },
    { 0x8B, "FAT32 mirrored" },
    { 0x8C, "FAT32 (LBA) mirrored" },
    { 0x8E, "Linux LVM" },
    { 0x93, "Hidden Linux or Amoeba" },
    { 0x94, "Amoeba Bad Block Table" },
    { 0x96, "CDFS/ISO-9660" },
    { 0x9F, "BSD/OS" },
    { 0xA0, "Laptop Hibernate" },                           // PARTITION_DIAGNOSTIC - "Diagnostic partition on some HP notebooks"
    { 0xA1, "Laptop Hibernate (NEC 6000H)" },
    { 0xA5, "BSD, NetBSD, FreeBSD" },
    { 0xA6, "OpenBSD" },
    { 0xA7, "NeXTStep" },
    { 0xA8, "Darwin UFS" },      // Also known as "MacOS-X"
    { 0xA9, "NetBSD" },
    { 0xAB, "Darwin boot" },
    { 0xAF, "Apple HFS/HFS+" },
    { 0xB1, "QNX6.x" },
    { 0xB2, "QNX6.x" },
    { 0xB3, "QNX6.x" },
    { 0xB6, "NT FAT16 corrupt mirror" },
    { 0xB7, "BSDI BSD/386 FS" }, // Alternatively, "NT NTFS corrupt mirror"
    { 0xB8, "BSDI BSD/386 swap" },
    { 0xBB, "Boot Wizard hidden" },
    { 0xBC, "Paragon Backup capsule" },
    { 0xBE, "Solaris 8 boot partition" },
    { 0xBF, "Solaris 10 x86" },
    { 0xC0, "NTFT" },            // Alternatively, "CTOS" or "REAL/32 or DR-DOS or Novell-DOS secure partition"
    { 0xC1, "DR-DOS FAT12" },
    { 0xC2, "Hidden Linux" },
    { 0xC3, "Hidden Linux swap" },
    { 0xC4, "DR-DOS FAT16 (< 32 MB)" },
    { 0xC5, "DR-DOS Extended" },
    { 0xC6, "DR-DOS FAT16" },
    { 0xC7, "HPFS mirrored" },   // Alternatively, "Syrinx boot"
    { 0xCB, "DR-DOS FAT32" },
    { 0xCC, "DR-DOS FAT32 (LBA)" },
    { 0xCE, "DR-DOS FAT16 (LBA)" },
    { 0xD0, "MDOS" },
    { 0xD1, "MDOS FAT12" },
    { 0xD4, "MDOS FAT16 (< 32 MB)" },
    { 0xD5, "MDOS Extended" },
    { 0xD6, "MDOS FAT16" },
    { 0xD7, "Microsoft Storage Spaces Data" },              // PARTITION_SPACES_DATA - "Storage Spaces protective partition"
    { 0xD8, "CP/M-86" },
    { 0xDB, "Digital Research CP/M" },
    { 0xDE, "Dell OEM" },                                   // PARTITION_DELL - "Dell partition"
    { 0xDF, "BootIt EMBRM (FAT16/32)" },
    { 0xE1, "SpeedStor FAT12" },
    { 0xE3, "SpeedStor (0xE3)" },
    { 0xE4, "SpeedStor FAT16" },
    { 0xE5, "Tandy MSDOS" },
    { 0xE6, "SpeedStor (0xE6)" },
    { 0xE7, "Microsoft Storage Spaces Protective" },        // PARTITION_SPACES - "Storage Spaces protective partition"
    { 0xE8, "Linux Unified Key Setup partition" },
    { 0xEA, "Rufus private partition" },
    { 0xEB, "BeOS/Haiku BFS" },
    { 0xEC, "SkyOS SkyFS" },
    { 0xEE, "EFI GPT protective" },                         // PARTITION_GPT    - "GPT protective partition"
    { 0xEF, "EFI System partition" },                       // PARTITION_SYSTEM - "System partition"
    { 0xF0, "Linux/PA-RISC boot loader" },
    { 0xF1, "SpeedStor (0xF1)" },
    { 0xF2, "DOS 3.3+ second" },
    { 0xF4, "SpeedStor (0xF4)" },
    { 0xF5, "SpeedStor (0xF5)" },
    { 0xF6, "SpeedStor (0xF6)" },
    { 0xFA, "Bochs" },
    { 0xFB, "VMware FS" },
    { 0xFC, "VMware swap/VMKCORE" },
    { 0xFD, "Linux RAID auto" },
    { 0xFE, "NT hidden or IBM PS/2 IML or Veritas VM" },    // PARTITION_IBM - "IBM IML partition"
    { 0xFF, "XENIX Bad Block Table or Veritas VM" },
};


/* GPT PARTITION TYPES ******************************************************/

#define GUID_CONST(l, w1, w2, b1, b2, b3, b4, b5, b6, b7, b8) \
    { l, w1, w2, { b1, b2, b3, b4, b5, b6, b7, b8 } }

#if 0
//
// NOTE: For information only: The uncommented GUIDs below
// are new ones defined in diskguid.h in Windows 10.
//

/* Partition GUIDs */
#include <initguid.h>
#include <diskguid.h>

/*
 * EFI specification
 */
// DEFINE_GUID(PARTITION_ENTRY_UNUSED_GUID,     0x00000000, 0x0000, 0x0000, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00);    // Entry unused
// DEFINE_GUID(PARTITION_SYSTEM_GUID,           0xC12A7328, 0xF81F, 0x11D2, 0xBA, 0x4B, 0x00, 0xA0, 0xC9, 0x3E, 0xC9, 0x3B);    // EFI system partition
DEFINE_GUID(PARTITION_LEGACYMBR_GUID,        0x024DEE41, 0x33E7, 0x11D3, 0x9D, 0x69, 0x00, 0x08, 0xC7, 0x81, 0xF3, 0x9F);

/*
 * Microsoft Windows / ReactOS
 */
// DEFINE_GUID(PARTITION_BASIC_DATA_GUID,       0xEBD0A0A2, 0xB9E5, 0x4433, 0x87, 0xC0, 0x68, 0xB6, 0xB7, 0x26, 0x99, 0xC7);    // Basic data partition
DEFINE_GUID(PARTITION_MAIN_OS_GUID,          0x57434F53, 0x8F45, 0x405E, 0x8A, 0x23, 0x18, 0x6D, 0x8A, 0x43, 0x30, 0xD3);    // Main OS partition
DEFINE_GUID(PARTITION_OS_DATA_GUID,          0x57434F53, 0x23F2, 0x44D5, 0xA8, 0x30, 0x67, 0xBB, 0xDA, 0xA6, 0x09, 0xF9);    // OS data partition
DEFINE_GUID(PARTITION_LEGACY_BL_GUID,        0x424CA0E2, 0x7CB2, 0x4FB9, 0x81, 0x43, 0xC5, 0x2A, 0x99, 0x39, 0x8B, 0xC6);    // Legacy boot loader partition
DEFINE_GUID(PARTITION_LEGACY_BL_GUID_BACKUP, 0x424C3E6C, 0xD79F, 0x49CB, 0x93, 0x5D, 0x36, 0xD7, 0x14, 0x67, 0xA2, 0x88);    // Legacy boot loader backup partition
/** Logical Disk Manager (LDM) **/
// DEFINE_GUID(PARTITION_LDM_METADATA_GUID,     0x5808C8AA, 0x7E8F, 0x42E0, 0x85, 0xD2, 0xE1, 0xE9, 0x04, 0x34, 0xCF, 0xB3);    // Logical Disk Manager metadata partition
// DEFINE_GUID(PARTITION_LDM_DATA_GUID,         0xAF9B60A0, 0x1431, 0x4F62, 0xBC, 0x68, 0x33, 0x11, 0x71, 0x4A, 0x69, 0xAD);    // Logical Disk Manager data partition
// DEFINE_GUID(PARTITION_MSFT_RESERVED_GUID,    0xE3C9E316, 0x0B5C, 0x4DB8, 0x81, 0x7D, 0xF9, 0x2D, 0xF0, 0x02, 0x15, 0xAE);    // Microsoft reserved space
DEFINE_GUID(PARTITION_MSFT_RECOVERY_GUID,    0xDE94BBA4, 0x06D1, 0x4D40, 0xA1, 0x6A, 0xBF, 0xD5, 0x01, 0x79, 0xD6, 0xAC);    // Microsoft recovery partition
DEFINE_GUID(PARTITION_MSFT_SNAPSHOT_GUID,    0xCADDEBF1, 0x4400, 0x4DE8, 0xB1, 0x03, 0x12, 0x11, 0x7D, 0xCF, 0x3C, 0xCF);    // Microsoft shadow copy partition
DEFINE_GUID(PARTITION_CLUSTER_GUID,          0xDB97DBA9, 0x0840, 0x4BAE, 0x97, 0xF0, 0xFF, 0xB9, 0xA3, 0x27, 0xC7, 0xE1);    // Cluster metadata partition
/** Storage Spaces & Replica **/
DEFINE_GUID(PARTITION_SPACES_GUID,           0xE75CAF8F, 0xF680, 0x4CEE, 0xAF, 0xA3, 0xB0, 0x01, 0xE5, 0x6E, 0xFC, 0x2D);    // Storage Spaces protective partition
DEFINE_GUID(PARTITION_SPACES_DATA_GUID,      0xE7ADDCB4, 0xDC34, 0x4539, 0x9A, 0x76, 0xEB, 0xBD, 0x07, 0xBE, 0x6F, 0x7E);    // Storage Spaces protective partition
DEFINE_GUID(PARTITION_STORAGE_REPLICA_GUID,  0x558D43C5, 0xA1AC, 0x43C0, 0xAA, 0xC8, 0xD1, 0x47, 0x2B, 0x29, 0x23, 0xD1);
DEFINE_GUID(PARTITION_PATCH_GUID,            0x8967A686, 0x96AA, 0x6AA8, 0x95, 0x89, 0xA8, 0x42, 0x56, 0x54, 0x10, 0x90);    // Patch partition
DEFINE_GUID(PARTITION_PRE_INSTALLED_GUID,    0x57434F53, 0x7FE0, 0x4196, 0x9B, 0x42, 0x42, 0x7B, 0x51, 0x64, 0x34, 0x84);    // PreInstalled partition
DEFINE_GUID(PARTITION_WINDOWS_SYSTEM_GUID,   0x57434F53, 0xE3E3, 0x4631, 0xA5, 0xC5, 0x26, 0xD2, 0x24, 0x38, 0x73, 0xAA);    // Windows system partition
/** IoT partitions **/
DEFINE_GUID(PARTITION_BSP_GUID,              0x57434F53, 0x4DF9, 0x45B9, 0x8E, 0x9E, 0x23, 0x70, 0xF0, 0x06, 0x45, 0x7C);    // BSP partition
DEFINE_GUID(PARTITION_DPP_GUID,              0x57434F53, 0x94CB, 0x43F0, 0xA5, 0x33, 0xD7, 0x3C, 0x10, 0xCF, 0xA5, 0x7D);    // Device provisioning (DPP) partition

#endif

/*
 * This partition type GUIDs list is based from:
 * - the kernelDisk.c and gpt.h modules of the Visopsys Operating System
 *   (see license above),
 * - NetBSD (Copyright (c) 2002 Marcel Moolenaar),
 *   http://cvsweb.netbsd.org/bsdweb.cgi/~checkout~/src/sys/sys/disklabel_gpt.h?rev=1.14&content-type=text/plain&only_with_tag=MAIN
 * - Haiku OS (Copyright 2003-2011, Haiku, Inc., under the terms of the MIT License)
 *   https://git.haiku-os.org/haiku/tree/src/add-ons/kernel/partitioning_systems/gpt/gpt_known_guids.h#n56
 * - and the following websites:
 *   https://www.liquisearch.com/guid_partition_table/partition_type_guids
 *   https://en.wikipedia.org/wiki/GUID_Partition_Table#Partition_type_GUIDs
 *   https://www.freedesktop.org/wiki/Specifications/DiscoverablePartitionsSpec/
 *   https://systemd.io/DISCOVERABLE_PARTITIONS/
 *   https://www.magnumdb.com/search?q=PARTITION_*
 */

/* Known GPT partition type GUIDs and descriptions */
const GPT_PARTITION_TYPE GptPartitionTypes[NUM_GPT_PARTITION_TYPES] =
{
    /*
     * EFI specification
     */
    { // PARTITION_ENTRY_UNUSED_GUID,
      GUID_CONST(0x00000000, 0x0000, 0x0000, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00),
        "Unused partition" }, // "Unused entry"
    { // PARTITION_SYSTEM_GUID,
      GUID_CONST(0xC12A7328, 0xF81F, 0x11D2, 0xBA, 0x4B, 0x00, 0xA0, 0xC9, 0x3E, 0xC9, 0x3B),
        "EFI/GPT System Partition" }, // "GPT Free Block"
    { // PARTITION_LEGACYMBR_GUID,
      GUID_CONST(0x024DEE41, 0x33E7, 0x11D3, 0x9D, 0x69, 0x00, 0x08, 0xC7, 0x81, 0xF3, 0x9F),
        "Legacy Master Boot Record Partition" },

    /*
     * Various
     */
    // GRUB 2 - "Hah!IdontNeedEFI"
    { GUID_CONST(0x21686148, 0x6449, 0x6E6F, 0x74, 0x4E, 0x65, 0x65, 0x64, 0x45, 0x46, 0x49),
        "GRUB 2 BIOS Boot partition" },
    // Intel Fast Flash (iFFS) partition (for Intel Rapid Start technology)
    { GUID_CONST(0xD3BFE2DE, 0x3DAF, 0x11DF, 0xBA, 0x40, 0xE3, 0xA5, 0x56, 0xD8, 0x95, 0x93),
        "Intel Fast Flash (iFFS) partition" },
    { GUID_CONST(0xF4019732, 0x066E, 0x4E12, 0x82, 0x73, 0x34, 0x6C, 0x56, 0x41, 0x49, 0x4F),
        "Sony boot partition" },
    { GUID_CONST(0xBFBFAFE7, 0xA34F, 0x448A, 0x9A, 0x5B, 0x62, 0x13, 0xEB, 0x73, 0x6C, 0x22),
        "Lenovo boot partition" },

    /*
     * Microsoft Windows / ReactOS
     */
    { // PARTITION_BASIC_DATA_GUID,
      GUID_CONST(0xEBD0A0A2, 0xB9E5, 0x4433, 0x87, 0xC0, 0x68, 0xB6, 0xB7, 0x26, 0x99, 0xC7),
        "Microsoft Basic Data partition" },
    { // PARTITION_MAIN_OS_GUID,
      GUID_CONST(0x57434F53, 0x8F45, 0x405E, 0x8A, 0x23, 0x18, 0x6D, 0x8A, 0x43, 0x30, 0xD3),
        "Main OS partition" },
    { // PARTITION_OS_DATA_GUID,
      GUID_CONST(0x57434F53, 0x23F2, 0x44D5, 0xA8, 0x30, 0x67, 0xBB, 0xDA, 0xA6, 0x09, 0xF9),
        "OS Data partition" },
    { // PARTITION_LEGACY_BL_GUID,
      GUID_CONST(0x424CA0E2, 0x7CB2, 0x4FB9, 0x81, 0x43, 0xC5, 0x2A, 0x99, 0x39, 0x8B, 0xC6),
        "Microsoft Legacy Boot Loader partition" },
    { // PARTITION_LEGACY_BL_GUID_BACKUP,
      GUID_CONST(0x424C3E6C, 0xD79F, 0x49CB, 0x93, 0x5D, 0x36, 0xD7, 0x14, 0x67, 0xA2, 0x88),
        "Microsoft Legacy Boot Loader backup partition" },
    { // PARTITION_LDM_METADATA_GUID,
      GUID_CONST(0x5808C8AA, 0x7E8F, 0x42E0, 0x85, 0xD2, 0xE1, 0xE9, 0x04, 0x34, 0xCF, 0xB3),
        "Microsoft LDM MetaData partition" },
    { // PARTITION_LDM_DATA_GUID,
      GUID_CONST(0xAF9B60A0, 0x1431, 0x4F62, 0xBC, 0x68, 0x33, 0x11, 0x71, 0x4A, 0x69, 0xAD),
        "Microsoft LDM Data partition" },
    { // PARTITION_MSFT_RESERVED_GUID,
      GUID_CONST(0xE3C9E316, 0x0B5C, 0x4DB8, 0x81, 0x7D, 0xF9, 0x2D, 0xF0, 0x02, 0x15, 0xAE),
        "Microsoft Reserved partition" },
    { // PARTITION_MSFT_RECOVERY_GUID,
      GUID_CONST(0xDE94BBA4, 0x06D1, 0x4D40, 0xA1, 0x6A, 0xBF, 0xD5, 0x01, 0x79, 0xD6, 0xAC),
        "Microsoft Recovery partition" },
    { // PARTITION_MSFT_SNAPSHOT_GUID,
      GUID_CONST(0xCADDEBF1, 0x4400, 0x4DE8, 0xB1, 0x03, 0x12, 0x11, 0x7D, 0xCF, 0x3C, 0xCF),
        "Microsoft Shadow Copy partition" },
    { // PARTITION_CLUSTER_GUID,
      GUID_CONST(0xDB97DBA9, 0x0840, 0x4BAE, 0x97, 0xF0, 0xFF, 0xB9, 0xA3, 0x27, 0xC7, 0xE1),
        "Microsoft Cluster MetaData partition" },
    { // PARTITION_SPACES_GUID,
      GUID_CONST(0xE75CAF8F, 0xF680, 0x4CEE, 0xAF, 0xA3, 0xB0, 0x01, 0xE5, 0x6E, 0xFC, 0x2D),
        "Microsoft Storage Spaces partition" },
    { // PARTITION_SPACES_DATA_GUID,
      GUID_CONST(0xE7ADDCB4, 0xDC34, 0x4539, 0x9A, 0x76, 0xEB, 0xBD, 0x07, 0xBE, 0x6F, 0x7E),
        "Microsoft Storage Spaces Data" },
    { // PARTITION_STORAGE_REPLICA_GUID,
      GUID_CONST(0x558D43C5, 0xA1AC, 0x43C0, 0xAA, 0xC8, 0xD1, 0x47, 0x2B, 0x29, 0x23, 0xD1),
        "Microsoft Storage Replica partition" },
    { // PARTITION_PATCH_GUID,
      GUID_CONST(0x8967A686, 0x96AA, 0x6AA8, 0x95, 0x89, 0xA8, 0x42, 0x56, 0x54, 0x10, 0x90),
        "Microsoft Patch partition" },
    { // PARTITION_PRE_INSTALLED_GUID,
      GUID_CONST(0x57434F53, 0x7FE0, 0x4196, 0x9B, 0x42, 0x42, 0x7B, 0x51, 0x64, 0x34, 0x84),
        "Microsoft PreInstalled partition" },
    { // PARTITION_WINDOWS_SYSTEM_GUID,
      GUID_CONST(0x57434F53, 0xE3E3, 0x4631, 0xA5, 0xC5, 0x26, 0xD2, 0x24, 0x38, 0x73, 0xAA),
        "Windows System partition" },
    { // PARTITION_BSP_GUID,
      GUID_CONST(0x57434F53, 0x4DF9, 0x45B9, 0x8E, 0x9E, 0x23, 0x70, 0xF0, 0x06, 0x45, 0x7C),
        "BSP partition" },
    { // PARTITION_DPP_GUID,
      GUID_CONST(0x57434F53, 0x94CB, 0x43F0, 0xA5, 0x33, 0xD7, 0x3C, 0x10, 0xCF, 0xA5, 0x7D),
        "Device provisioning (DPP) partition" },

    /*
     * IBM
     */
    // "IBM General Parallel File System (GPFS) partition"
    { GUID_CONST(0x37AFFC90, 0xEF7D, 0x4E96, 0x91, 0xC3, 0x2D, 0x7A, 0xE0, 0x55, 0xB1, 0x74),
        "IBM GPFS" },
    { GUID_CONST(0x9E1A2D38, 0xC612, 0x4316, 0xAA, 0x26, 0x8B, 0x49, 0x52, 0x1E, 0x5A, 0x8B),
        "IBM PowerPC PReP boot" },

    /*
     * HP-UX
     */
    { GUID_CONST(0x75894C1E, 0x3AEB, 0x11D3, 0xB7, 0xC1, 0x7B, 0x03, 0xA0, 0x00, 0x00, 0x00),
        "HP-UX Data partition" },
    { GUID_CONST(0xE2A1E728, 0x32E3, 0x11D6, 0xA6, 0x82, 0x7B, 0x03, 0xA0, 0x00, 0x00, 0x00),
        "HP-UX Service partition" },

    /*
     * freedesktop.org / systemd
     */
    /* Root partitions */
    { GUID_CONST(0x44479540, 0xF297, 0x41B2, 0x9A, 0xF7, 0xD1, 0x31, 0xD5, 0xF0, 0x45, 0x8A),
        "Freedesktop Root partition (x86)" },
    { GUID_CONST(0x4F68BCE3, 0xE8CD, 0x4DB1, 0x96, 0xE7, 0xFB, 0xCA, 0xF9, 0x84, 0xB7, 0x09),
        "Freedesktop Root partition (x86-64)" },
    { GUID_CONST(0x69DAD710, 0x2CE4, 0x4E3C, 0xB1, 0x6C, 0x21, 0xA1, 0xD4, 0x9A, 0xBE, 0xD3),
        "Freedesktop Root partition (32-bit ARM)" },
    { GUID_CONST(0xB921B045, 0x1DF0, 0x41C3, 0xAF, 0x44, 0x4C, 0x6F, 0x28, 0x0D, 0x3F, 0xAE),
        "Freedesktop Root partition (64-bit ARM/AArch64)" },
    { GUID_CONST(0x993D8D3D, 0xF80E, 0x4225, 0x85, 0x5A, 0x9D, 0xAF, 0x8E, 0xD7, 0xEA, 0x97),
        "Freedesktop Root partition (IA64)" },
    /* Root dm-verity partitions */
    { GUID_CONST(0xD13C5D3B, 0xB5D1, 0x422A, 0xB2, 0x9F, 0x94, 0x54, 0xFD, 0xC8, 0x9D, 0x76),
        "Freedesktop Root verity partition (x86)" },
    { GUID_CONST(0x2C7357ED, 0xEBD2, 0x46D9, 0xAE, 0xC1, 0x23, 0xD4, 0x37, 0xEC, 0x2B, 0xF5),
        "Freedesktop Root verity partition (x86-64)" },
    { GUID_CONST(0x7386CDF2, 0x203C, 0x47A9, 0xA4, 0x98, 0xF2, 0xEC, 0xCE, 0x45, 0xA2, 0xD6),
        "Freedesktop Root verity partition (32-bit ARM)" },
    { GUID_CONST(0xDF3300CE, 0xD69F, 0x4C92, 0x97, 0x8C, 0x9B, 0xFB, 0x0F, 0x38, 0xD8, 0x20),
        "Freedesktop Root verity partition (64-bit ARM/AArch64)" },
    { GUID_CONST(0x86ED10D5, 0xB607, 0x45BB, 0x89, 0x57, 0xD3, 0x50, 0xF2, 0x3D, 0x05, 0x71),
        "Freedesktop Root verity partition (IA64)" },
    /* User partitions */
    { GUID_CONST(0x75250D76, 0x8CC6, 0x458E, 0xBD, 0x66, 0xBD, 0x47, 0xCC, 0x81, 0xA8, 0x12),
        "Freedesktop /usr partition (x86)" },
    { GUID_CONST(0x8484680C, 0x9521, 0x48C6, 0x9C, 0x11, 0xB0, 0x72, 0x06, 0x56, 0xF6, 0x9E),
        "Freedesktop /usr partition (x86-64)" },
    { GUID_CONST(0x7D0359A3, 0x02B3, 0x4F0A, 0x86, 0x5C, 0x65, 0x44, 0x03, 0xE7, 0x06, 0x25),
        "Freedesktop /usr partition (32-bit ARM)" },
    { GUID_CONST(0xB0E01050, 0xEE5F, 0x4390, 0x94, 0x9A, 0x91, 0x01, 0xB1, 0x71, 0x04, 0xE9),
        "Freedesktop /usr partition (64-bit ARM/AArch64)" },
    { GUID_CONST(0x4301D2A6, 0x4E3B, 0x4B2A, 0xBB, 0x94, 0x9E, 0x0B, 0x2C, 0x42, 0x25, 0xEA),
        "Freedesktop /usr partition (IA64)" },
    /* User dm-verity partitions */
    { GUID_CONST(0x8F461B0D, 0x14EE, 0x4E81, 0x9A, 0xA9, 0x04, 0x9B, 0x6F, 0xB9, 0x7A, 0xBD),
        "Freedesktop /usr verity partition (x86)" },
    { GUID_CONST(0x77FF5F63, 0xE7B6, 0x4633, 0xAC, 0xF4, 0x15, 0x65, 0xB8, 0x64, 0xC0, 0xE6),
        "Freedesktop /usr verity partition (x86-64)" },
    { GUID_CONST(0xC215D751, 0x7BCD, 0x4649, 0xBE, 0x90, 0x66, 0x27, 0x49, 0x0A, 0x4C, 0x05),
        "Freedesktop /usr verity partition (32-bit ARM)" },
    { GUID_CONST(0x6E11A4E7, 0xFBCA, 0x4DED, 0xB9, 0xE9, 0xE1, 0xA5, 0x12, 0xBB, 0x66, 0x4E),
        "Freedesktop /usr verity partition (64-bit ARM/AArch64)" },
    { GUID_CONST(0x6A491E03, 0x3BE7, 0x4545, 0x8E, 0x38, 0x83, 0x32, 0x0E, 0x0E, 0xA8, 0x80),
        "Freedesktop /usr verity partition (IA64)" },
    /* Native */
    { GUID_CONST(0x933AC7E1, 0x2EB4, 0x4F13, 0xB8, 0x44, 0x0E, 0x14, 0xE2, 0xAE, 0xF9, 0x15),
        "Freedesktop /home partition" },
    { GUID_CONST(0x3B8F8425, 0x20E0, 0x4F3B, 0x90, 0x7F, 0x1A, 0x25, 0xA7, 0x6F, 0x98, 0xE8),
        "Freedesktop /srv (server data) partition" },
    /* Variable data */
    { GUID_CONST(0x4D21B016, 0xB534, 0x45C2, 0xA9, 0xFB, 0x5C, 0x16, 0xE0, 0x91, 0xFD, 0x2D),
        "Freedesktop /var partition" },
    { GUID_CONST(0x7EC6F557, 0x3BC5, 0x4ACA, 0xB2, 0x93, 0x16, 0xEF, 0x5D, 0xF6, 0x39, 0xD1),
        "Freedesktop /var/tmp partition" },
    /* Boot partition */
    // "Extended Boot Loader Partition" / "Shared boot loader configuration"
    { GUID_CONST(0xBC13C2FF, 0x59E6, 0x4262, 0xA3, 0x52, 0xB2, 0x75, 0xFD, 0x6F, 0x71, 0x72),
        "Freedesktop /boot partition" },

    /*
     * Linux
     */
    { GUID_CONST(0x0FC63DAF, 0x8483, 0x4772, 0x8E, 0x79, 0x3D, 0x69, 0xD8, 0x47, 0x7D, 0xE4),
        "Linux data" },
    { GUID_CONST(0xA19D880F, 0x05FC, 0x4D3B, 0xA0, 0x06, 0x74, 0x3F, 0x0F, 0x84, 0x91, 0x1E),
        "Linux RAID partition" },
    { GUID_CONST(0x0657FD6D, 0xA4AB, 0x43C4, 0x84, 0xE5, 0x09, 0x33, 0xC8, 0x4B, 0x4F, 0x4F),
        "Linux Swap partition" },
    // Logical Volume Manager (LVM)
    { GUID_CONST(0xE6D6D379, 0xF507, 0x44C2, 0xA2, 0x3C, 0x23, 0x8F, 0x2A, 0x3D, 0xF9, 0x28),
        "Linux LVM partition" },
    { GUID_CONST(0x8DA63339, 0x0007, 0x60C0, 0xC4, 0x36, 0x08, 0x3A, 0xC8, 0x23, 0x09, 0x08),
        "Linux reserved" },
    { GUID_CONST(0x7FFEC5C9, 0x2D00, 0x49B7, 0x89, 0x41, 0x3E, 0xA1, 0x0A, 0x55, 0x86, 0xB7),
        "Linux Plain dm-crypt partition" },
    { GUID_CONST(0xCA7D7CCB, 0x63ED, 0x4C53, 0x86, 0x1C, 0x17, 0x42, 0x53, 0x60, 0x59, 0xCC),
        "Linux LUKS partition" },

    /*
     * FreeBSD
     */
    { GUID_CONST(0x83BD6B9D, 0x7F41, 0x11DC, 0xBE, 0x0B, 0x00, 0x15, 0x60, 0xB8, 0x4F, 0x0F),
        "FreeBSD Boot" },
    { GUID_CONST(0x516E7CB4, 0x6ECF, 0x11D6, 0x8F, 0xF8, 0x00, 0x02, 0x2D, 0x09, 0x71, 0x2B),
        "FreeBSD Data" },
    { GUID_CONST(0x516E7CB5, 0x6ECF, 0x11D6, 0x8F, 0xF8, 0x00, 0x02, 0x2D, 0x09, 0x71, 0x2B),
        "FreeBSD Swap" },
    // Unix File System (UFS)
    { GUID_CONST(0x516E7CB6, 0x6ECF, 0x11D6, 0x8F, 0xF8, 0x00, 0x02, 0x2D, 0x09, 0x71, 0x2B),
        "FreeBSD UFS" },
    /* The following is unused but documented here to avoid reuse.
     * GUID_CONST(0x516E7CB7, 0x6ECF, 0x11D6, 0x8F, 0xF8, 0x00, 0x02, 0x2D, 0x09, 0x71, 0x2B);
     */
    // Vinum volume manager
    { GUID_CONST(0x516E7CB8, 0x6ECF, 0x11D6, 0x8F, 0xF8, 0x00, 0x02, 0x2D, 0x09, 0x71, 0x2B),
        "FreeBSD Vinum" },
    { GUID_CONST(0x516E7CBA, 0x6ECF, 0x11D6, 0x8F, 0xF8, 0x00, 0x02, 0x2D, 0x09, 0x71, 0x2B),
        "FreeBSD ZFS" },

    /*
     * MidnightBSD
     */
    { GUID_CONST(0x85D5E45E, 0x237C, 0x11E1, 0xB4, 0xB3, 0xE8, 0x9A, 0x8F, 0x7F, 0xC3, 0xA7),
        "MidnightBSD Boot" },
    { GUID_CONST(0x85D5E45A, 0x237C, 0x11E1, 0xB4, 0xB3, 0xE8, 0x9A, 0x8F, 0x7F, 0xC3, 0xA7),
        "MidnightBSD Data" },
    { GUID_CONST(0x85D5E45B, 0x237C, 0x11E1, 0xB4, 0xB3, 0xE8, 0x9A, 0x8F, 0x7F, 0xC3, 0xA7),
        "MidnightBSD Swap" },
    // Unix File System (UFS)
    { GUID_CONST(0x0394EF8B, 0x237E, 0x11E1, 0xB4, 0xB3, 0xE8, 0x9A, 0x8F, 0x7F, 0xC3, 0xA7),
        "MidnightBSD UFS" },
    // Vinum volume manager
    { GUID_CONST(0x85D5E45C, 0x237C, 0x11E1, 0xB4, 0xB3, 0xE8, 0x9A, 0x8F, 0x7F, 0xC3, 0xA7),
        "MidnightBSD Vinum" },
    { GUID_CONST(0x85D5E45D, 0x237C, 0x11E1, 0xB4, 0xB3, 0xE8, 0x9A, 0x8F, 0x7F, 0xC3, 0xA7),
        "MidnightBSD ZFS" },

    /*
     * NetBSD
     */
    { GUID_CONST(0x49F48D32, 0xB10E, 0x11DC, 0xB9, 0x9B, 0x00, 0x19, 0xD1, 0x87, 0x96, 0x48),
        "NetBSD Swap" },
    { GUID_CONST(0x49F48D5A, 0xB10E, 0x11DC, 0xB9, 0x9B, 0x00, 0x19, 0xD1, 0x87, 0x96, 0x48),
        "NetBSD FFS" },
    { GUID_CONST(0x49F48D82, 0xB10E, 0x11DC, 0xB9, 0x9B, 0x00, 0x19, 0xD1, 0x87, 0x96, 0x48),
        "NetBSD LFS" },
    // NetBSD RAID Frame
    { GUID_CONST(0x49F48DAA, 0xB10E, 0x11DC, 0xB9, 0x9B, 0x00, 0x19, 0xD1, 0x87, 0x96, 0x48),
        "NetBSD RAID" },
    // NetBSD CCD
    { GUID_CONST(0x2DB519C4, 0xB10F, 0x11DC, 0xB9, 0x9B, 0x00, 0x19, 0xD1, 0x87, 0x96, 0x48),
        "NetBSD Concatenated partition" },
    // NetBSD CGD
    { GUID_CONST(0x2DB519EC, 0xB10F, 0x11DC, 0xB9, 0x9B, 0x00, 0x19, 0xD1, 0x87, 0x96, 0x48),
        "NetBSD Encrypted partition" },

    /*
     * OpenBSD
     */
    { GUID_CONST(0x824CC7A0, 0x36A8, 0x11E3, 0x89, 0x0A, 0x95, 0x25, 0x19, 0xAD, 0x3F, 0x61),
        "OpenBSD Data" },

    /*
     * DragonFlyBSD
     */
    { GUID_CONST(0x9D087404, 0x1CA5, 0x11DC, 0x88, 0x17, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD L32" },
    { GUID_CONST(0x3D48CE54, 0x1D16, 0x11DC, 0x86, 0x96, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD L64" },
    { GUID_CONST(0xBD215AB2, 0x1D16, 0x11DC, 0x86, 0x96, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD Legacy" },
    { GUID_CONST(0xDBD5211B, 0x1CA5, 0x11DC, 0x88, 0x17, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD Concatenated partition" },
    { GUID_CONST(0x61DC63AC, 0x6E38, 0x11DC, 0x85, 0x13, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD Hammer" },
    { GUID_CONST(0x5CBB9AD1, 0x862D, 0x11DC, 0xA9, 0x4D, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD Hammer 2" },
    { GUID_CONST(0x9D58FDBD, 0x1CA5, 0x11DC, 0x88, 0x17, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD Swap" },
    // Unix File System (UFS)
    { GUID_CONST(0x9D94CE7C, 0x1CA5, 0x11DC, 0x88, 0x17, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD UFS" },
    // Vinum volume manager
    { GUID_CONST(0x9DD4478F, 0x1CA5, 0x11DC, 0x88, 0x17, 0x01, 0x30, 0x1B, 0xB8, 0xA9, 0xF5),
        "DragonFlyBSD Vinum" },

    /*
     * Apple / MacOS Darwin
     */
    { GUID_CONST(0x6A898CC3, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Apple ZFS / Solaris /usr" },
    // APPLE_HFS "Apple data partition" / "Mac OS X Hierarchical File System Plus (HFS+) partition"
    { GUID_CONST(0x48465300, 0x0000, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "MacOS X HFS+" },
    // "Apple APFS container"
    { GUID_CONST(0x7C3457EF, 0x0000, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "APFS FileVault volume container" },
    { GUID_CONST(0x55465300, 0x0000, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "Apple UFS container" },
    { GUID_CONST(0x52414944, 0x0000, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "Apple RAID" },
    { GUID_CONST(0x52414944, 0x5F4F, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "Apple RAID offline" },
    { GUID_CONST(0x426F6F74, 0x0000, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "Apple Boot partition (Recovery HD)" },
    { GUID_CONST(0x4C616265, 0x6C00, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "Apple Label" },
    { GUID_CONST(0x5265636F, 0x7665, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "Apple TV Recovery partition" },
    { GUID_CONST(0x53746F72, 0x6167, 0x11AA, 0xAA, 0x11, 0x00, 0x30, 0x65, 0x43, 0xEC, 0xAC),
        "Apple Core Storage (Lion FileVault) Container" },
    { GUID_CONST(0xB6FA30DA, 0x92D2, 0x4A9A, 0x96, 0xF1, 0x87, 0x1E, 0xC6, 0x48, 0x62, 0x00),
        "SoftRAID_Status" },
    { GUID_CONST(0x2E313465, 0x19B9, 0x463F, 0x81, 0x26, 0x8A, 0x79, 0x93, 0x77, 0x38, 0x01),
        "SoftRAID_Scratch" },
    { GUID_CONST(0xFA709C7E, 0x65B1, 0x4593, 0xBF, 0xD5, 0xE7, 0x1D, 0x61, 0xDE, 0x9B, 0x02),
        "SoftRAID_Volume" },
    { GUID_CONST(0xBBBA6DF5, 0xF46F, 0x4A89, 0x8F, 0x59, 0x87, 0x65, 0xB2, 0x72, 0x75, 0x03),
        "SoftRAID_Cache" },

    /*
     * Solaris / Illumos
     */
    { GUID_CONST(0x6A82CB45, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Boot" },
    { GUID_CONST(0x6A85CF4D, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Root" },
    { GUID_CONST(0x6A87C46F, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Swap" },
    { GUID_CONST(0x6A8B642B, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Backup" },
    // See MacOS "Apple ZFS partition"
    // { GUID_CONST(0x6A898CC3, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        // "Solaris /usr" },
    { GUID_CONST(0x6A8EF2E9, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris /var" },
    { GUID_CONST(0x6A90BA39, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris /home" },
    { GUID_CONST(0x6A9283A5, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Alternate sector" },
    { GUID_CONST(0x6A945A3B, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Reserved" },
    { GUID_CONST(0x6A9630D1, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Reserved" },
    { GUID_CONST(0x6A980767, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Reserved" },
    { GUID_CONST(0x6A96237F, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Reserved" },
    { GUID_CONST(0x6A8D2AC7, 0x1DD2, 0x11B2, 0x99, 0xA6, 0x08, 0x00, 0x20, 0x73, 0x66, 0x31),
        "Solaris Reserved" },

    /*
     * ChromeOS
     */
    { GUID_CONST(0xFE3A2A5D, 0x4F32, 0x41A7, 0xB7, 0x25, 0xAC, 0xCC, 0x32, 0x85, 0xA3, 0x09),
        "ChromeOS kernel" },
    { GUID_CONST(0x3CB8E202, 0x3B7E, 0x47DD, 0x8A, 0x3C, 0x7F, 0xF2, 0xA1, 0x3C, 0xFC, 0xEC),
        "ChromeOS rootfs" },
    { GUID_CONST(0x2E0A753D, 0x9E48, 0x43B0, 0x83, 0x37, 0xB1, 0x51, 0x92, 0xCB, 0x1B, 0x5E),
        "ChromeOS future use" },

    /*
     * Haiku
     */
    { GUID_CONST(0x42465331, 0x3BA3, 0x10F1, 0x80, 0x2A, 0x48, 0x61, 0x69, 0x6B, 0x75, 0x21),
        "Haiku BFS" },

    /*
     * Container Linux
     */
    { GUID_CONST(0x5DFBF5F4, 0x2848, 0x4BAC, 0xAA, 0x5E, 0x0D, 0x9A, 0x20, 0xB7, 0x45, 0xA6),
        "/usr partition (coreos-usr)" },
    { GUID_CONST(0x3884DD41, 0x8582, 0x4404, 0xB9, 0xA8, 0xE9, 0xB8, 0x4F, 0x2D, 0xF5, 0x0E),
        "Resizable rootfs (coreos-resize)" },
    { GUID_CONST(0xC95DC21A, 0xDF0E, 0x4340, 0x8D, 0x7B, 0x26, 0xCB, 0xFA, 0x9A, 0x03, 0xE0),
        "OEM customizations (coreos-reserved)" },
    { GUID_CONST(0xBE9067B9, 0xEA49, 0x4F15, 0xB4, 0xF6, 0xF3, 0x6F, 0x8C, 0x9E, 0x18, 0x18),
        "Root filesystem on RAID (coreos-root-raid)" },

    /*
     * Ceph
     */
    { GUID_CONST(0x45B0969E, 0x9B03, 0x4F30, 0xB4, 0xC6, 0xB4, 0xB8, 0x0C, 0xEF, 0xF1, 0x06),
        "Ceph Journal" },
    { GUID_CONST(0x45B0969E, 0x9B03, 0x4F30, 0xB4, 0xC6, 0x5E, 0xC0, 0x0C, 0xEF, 0xF1, 0x06),
        "dm-crypt journal" },
    { GUID_CONST(0x4FBD7E29, 0x9D25, 0x41B8, 0xAF, 0xD0, 0x06, 0x2C, 0x0C, 0xEF, 0xF0, 0x5D),
        "Ceph OSD" },
    { GUID_CONST(0x4FBD7E29, 0x9D25, 0x41B8, 0xAF, 0xD0, 0x5E, 0xC0, 0x0C, 0xEF, 0xF0, 0x5D),
        "dm-crypt OSD" },
    { GUID_CONST(0x89C57F98, 0x2FE5, 0x4DC0, 0x89, 0xC1, 0xF3, 0xAD, 0x0C, 0xEF, 0xF2, 0xBE),
        "Ceph Disk in creation" },
    { GUID_CONST(0x89C57F98, 0x2FE5, 0x4DC0, 0x89, 0xC1, 0x5E, 0xC0, 0x0C, 0xEF, 0xF2, 0xBE),
        "dm-crypt disk in creation" },
    { GUID_CONST(0xCAFECAFE, 0x9B03, 0x4F30, 0xB4, 0xC6, 0xB4, 0xB8, 0x0C, 0xEF, 0xF1, 0x06),
        "Ceph Block" },
    { GUID_CONST(0x30CD0809, 0xC2B2, 0x499C, 0x88, 0x79, 0x2D, 0x6B, 0x78, 0x52, 0x98, 0x76),
        "Ceph Block DB" },
    { GUID_CONST(0x5CE17FCE, 0x4087, 0x4169, 0xB7, 0xFF, 0x05, 0x6C, 0xC5, 0x84, 0x73, 0xF9),
        "Ceph Block write-ahead log" },
    { GUID_CONST(0xFB3AABF9, 0xD25F, 0x47CC, 0xBF, 0x5E, 0x72, 0x1D, 0x18, 0x16, 0x49, 0x6B),
        "Lockbox for dm-crypt keys" },
    { GUID_CONST(0x4FBD7E29, 0x8AE0, 0x4982, 0xBF, 0x9D, 0x5A, 0x8D, 0x86, 0x7A, 0xF5, 0x60),
        "Ceph Multipath OSD" },
    { GUID_CONST(0x45B0969E, 0x8AE0, 0x4982, 0xBF, 0x9D, 0x5A, 0x8D, 0x86, 0x7A, 0xF5, 0x60),
        "Ceph Multipath journal" },
    { GUID_CONST(0xCAFECAFE, 0x8AE0, 0x4982, 0xBF, 0x9D, 0x5A, 0x8D, 0x86, 0x7A, 0xF5, 0x60),
        "Ceph Multipath block" },
    { GUID_CONST(0x7F4A666A, 0x16F3, 0x47A2, 0x84, 0x45, 0x15, 0x2E, 0xF4, 0xD0, 0x3F, 0x6C),
        "Ceph Multipath block" },
    { GUID_CONST(0xEC6D6385, 0xE346, 0x45DC, 0xBE, 0x91, 0xDA, 0x2A, 0x7C, 0x8B, 0x32, 0x61),
        "Ceph Multipath block DB" },
    { GUID_CONST(0x01B41E1B, 0x002A, 0x453C, 0x9F, 0x17, 0x88, 0x79, 0x39, 0x89, 0xFF, 0x8F),
        "Ceph Multipath block write-ahead log" },
    { GUID_CONST(0xCAFECAFE, 0x9B03, 0x4F30, 0xB4, 0xC6, 0x5E, 0xC0, 0x0C, 0xEF, 0xF1, 0x06),
        "dm-crypt block" },
    { GUID_CONST(0x93B0052D, 0x02D9, 0x4D8A, 0xA4, 0x3B, 0x33, 0xA3, 0xEE, 0x4D, 0xFB, 0xC3),
        "dm-crypt block DB" },
    { GUID_CONST(0x306E8683, 0x4FE2, 0x4330, 0xB7, 0xC0, 0x00, 0xA9, 0x17, 0xC1, 0x69, 0x66),
        "dm-crypt block write-ahead log" },
    { GUID_CONST(0x45B0969E, 0x9B03, 0x4F30, 0xB4, 0xC6, 0x35, 0x86, 0x5C, 0xEF, 0xF1, 0x06),
        "dm-crypt LUKS journal" },
    { GUID_CONST(0xCAFECAFE, 0x9B03, 0x4F30, 0xB4, 0xC6, 0x35, 0x86, 0x5C, 0xEF, 0xF1, 0x06),
        "dm-crypt LUKS block" },
    { GUID_CONST(0x166418DA, 0xC469, 0x4022, 0xAD, 0xF4, 0xB3, 0x0A, 0xFD, 0x37, 0xF1, 0x76),
        "dm-crypt LUKS block DB" },
    { GUID_CONST(0x86A32090, 0x3647, 0x40B9, 0xBB, 0xBD, 0x38, 0xD8, 0xC5, 0x73, 0xAA, 0x86),
        "dm-crypt LUKS block write-ahead log" },
    { GUID_CONST(0x4FBD7E29, 0x9D25, 0x41B8, 0xAF, 0xD0, 0x35, 0x86, 0x5C, 0xEF, 0xF0, 0x5D),
        "dm-crypt LUKS OSD" },

    /*
     * QNX
     */
    { GUID_CONST(0xCEF5A9AD, 0x73BC, 0x4601, 0x89, 0xF3, 0xCD, 0xEE, 0xEE, 0xE3, 0x21, 0xA1),
        "Power-safe (QNX6) file system" },

    /*
     * Plan 9
     */
    { GUID_CONST(0xC91818F9, 0x8025, 0x47AF, 0x89, 0xD2, 0xF0, 0x30, 0xD7, 0x00, 0x0C, 0x2C),
        "Plan 9 partition" },

    /*
     * VMware ESX
     */
    { GUID_CONST(0xAA31E02A, 0x400F, 0x11DB, 0x95, 0x90, 0x00, 0x0C, 0x29, 0x11, 0xD1, 0xB8),
        "VMware VMFS" },
    { GUID_CONST(0x9D275380, 0x40AD, 0x11DB, 0xBF, 0x97, 0x00, 0x0C, 0x29, 0x11, 0xD1, 0xB8),
        "VMware VMKCORE" },
    { GUID_CONST(0x9198EFFC, 0x31C0, 0x11DB, 0x8F, 0x78, 0x00, 0x0C, 0x29, 0x11, 0xD1, 0xB8),
        "VMware Reserved" },
    { GUID_CONST(0x381CFCCC, 0x7288, 0x11E0, 0x92, 0xEE, 0x00, 0x0C, 0x29, 0x11, 0xD0, 0xB2),
        "VMware vSAN" },

    /*
     * Android-IA
     */
    { GUID_CONST(0x2568845D, 0x2332, 0x4675, 0xBC, 0x39, 0x8F, 0xA5, 0xA4, 0x74, 0x8D, 0x15),
        "Android Bootloader" },
    { GUID_CONST(0x114EAFFE, 0x1552, 0x4022, 0xB2, 0x6E, 0x9B, 0x05, 0x36, 0x04, 0xCF, 0x84),
        "Android Bootloader2" },
    { GUID_CONST(0x49A4D17F, 0x93A3, 0x45C1, 0xA0, 0xDE, 0xF5, 0x0B, 0x2E, 0xBE, 0x25, 0x99),
        "Android Boot" },
    { GUID_CONST(0x4177C722, 0x9E92, 0x4AAB, 0x86, 0x44, 0x43, 0x50, 0x2B, 0xFD, 0x55, 0x06),
        "Android Recovery" },
    { GUID_CONST(0xEF32A33B, 0xA409, 0x486C, 0x91, 0x41, 0x9F, 0xFB, 0x71, 0x1F, 0x62, 0x66),
        "Android Misc" },
    { GUID_CONST(0x20AC26BE, 0x20B7, 0x11E3, 0x84, 0xC5, 0x6C, 0xFD, 0xB9, 0x47, 0x11, 0xE9),
        "Android Metadata" },
    { GUID_CONST(0x38F428E6, 0xD326, 0x425D, 0x91, 0x40, 0x6E, 0x0E, 0xA1, 0x33, 0x64, 0x7C),
        "Android System" },
    { GUID_CONST(0xA893EF21, 0xE428, 0x470A, 0x9E, 0x55, 0x06, 0x68, 0xFD, 0x91, 0xA2, 0xD9),
        "Android Cache" },
    { GUID_CONST(0xDC76DDA9, 0x5AC1, 0x491C, 0xAF, 0x42, 0xA8, 0x25, 0x91, 0x58, 0x0C, 0x0D),
        "Android Data" },
    { GUID_CONST(0xEBC597D0, 0x2053, 0x4B15, 0x8B, 0x64, 0xE0, 0xAA, 0xC7, 0x5F, 0x4D, 0xB1),
        "Android Persistent" },
    { GUID_CONST(0xC5A0AEEC, 0x13EA, 0x11E5, 0xA1, 0xB1, 0x00, 0x1E, 0x67, 0xCA, 0x0C, 0x3C),
        "Android Vendor" },
    { GUID_CONST(0xBD59408B, 0x4514, 0x490D, 0xBF, 0x12, 0x98, 0x78, 0xD9, 0x63, 0xF3, 0x78),
        "Android Config" },
    { GUID_CONST(0x8F68CC74, 0xC5E5, 0x48DA, 0xBE, 0x91, 0xA0, 0xC8, 0xC1, 0x5E, 0x9C, 0x80),
        "Android Factory" },
    { GUID_CONST(0x9FDAA6EF, 0x4B3F, 0x40D2, 0xBA, 0x8D, 0xBF, 0xF1, 0x6B, 0xFB, 0x88, 0x7B),
        "Android Factory (alt)" },
    { GUID_CONST(0x767941D0, 0x2085, 0x11E3, 0xAD, 0x3B, 0x6C, 0xFD, 0xB9, 0x47, 0x11, 0xE9),
        "Android Fastboot / Tertiary" },
    { GUID_CONST(0xAC6D7924, 0xEB71, 0x4DF8, 0xB4, 0x8D, 0xE2, 0x67, 0xB2, 0x71, 0x48, 0xFF),
        "Android OEM" },

    /*
     * Android 6.0+ ARM
     */
    { GUID_CONST(0x19A710A2, 0xB3CA, 0x11E4, 0xB0, 0x26, 0x10, 0x60, 0x4B, 0x88, 0x9D, 0xCF),
        "Android Meta" },
    { GUID_CONST(0x193D1EA4, 0xB3CA, 0x11E4, 0xB0, 0x75, 0x10, 0x60, 0x4B, 0x88, 0x9D, 0xCF),
        "Android EXT" },

    /*
     * ONIE (Open Network Install Environment)
     */
    { GUID_CONST(0x7412F7D5, 0xA156, 0x4B13, 0x81, 0xDC, 0x86, 0x71, 0x74, 0x92, 0x93, 0x25),
        "ONIE Boot" },
    { GUID_CONST(0xD4E6E2CD, 0x4469, 0x46F3, 0xB5, 0xCB, 0x1B, 0xFF, 0x57, 0xAF, 0xC1, 0x49),
        "ONIE Config" },

    /*
     * Atari TOS
     */
    { GUID_CONST(0x734E5AFE, 0xF61A, 0x11E6, 0xBC, 0x64, 0x92, 0x36, 0x1F, 0x00, 0x26, 0x71),
        "Atari TOS Basic data partition (GEM, BGM, F32)" },

    /*
     * VeraCrypt
     */
    { GUID_CONST(0x8C8F8EFF, 0xAC95, 0x4770, 0x81, 0x4A, 0x21, 0x99, 0x4F, 0x2D, 0xBC, 0x8F),
        "VeraCrypt Encrypted data partition" },

    /*
     * OS/2 ArcaOS
     */
    { GUID_CONST(0x90B6FF38, 0xB98F, 0x4358, 0xA2, 0x1F, 0x48, 0xF3, 0x5B, 0x4A, 0x8A, 0xD3),
        "ArcaOS Type 1" },
};

/* EOF */
