/*
 * PROJECT:     ReactOS API tests
 * LICENSE:     MIT (https://spdx.org/licenses/MIT)
 * PURPOSE:     Tests for floor / floorf
 * COPYRIGHT:   Copyright 2021 Timo Kreuzer <timo.kreuzer@reactos.org>
 */

#if !defined(_CRTBLD) && !defined(_M_IX86)
#define _CRTBLD // we don't want inline floorf!
#endif
#include "math_helpers.h"

#ifdef _MSC_VER
#pragma function(floor)
#ifdef _M_AMD64
#pragma function(floorf)
#endif
#endif

static TESTENTRY_DBL s_floor_tests[] =
{
    /* Special values */
    { 0x0000000000000000 /*  0.000000000000000e+000 */, 0x0000000000000000 /*  0.000000000000000e+000 */ },
    { 0x8000000000000000 /* -0.000000000000000e+000 */, 0x8000000000000000 /* -0.000000000000000e+000 */ },
    { 0x7ff0000000000000 /*  1.#INF00000000000e+000 */, 0x7ff0000000000000 /*  1.#INF00000000000e+000 */ },
    { 0x7ff0000000000001 /*  1.#QNAN0000000000e+000 */, 0x7ff8000000000001 /*  1.#QNAN0000000000e+000 */ },
    { 0x7ff7ffffffffffff /*  1.#QNAN0000000000e+000 */, 0x7fffffffffffffff /*  1.#QNAN0000000000e+000 */ },
    { 0x7ff8000000000000 /*  1.#QNAN0000000000e+000 */, 0x7ff8000000000000 /*  1.#QNAN0000000000e+000 */ },
    { 0x7ff8000000000001 /*  1.#QNAN0000000000e+000 */, 0x7ff8000000000001 /*  1.#QNAN0000000000e+000 */ },
    { 0x7fffffffffffffff /*  1.#QNAN0000000000e+000 */, 0x7fffffffffffffff /*  1.#QNAN0000000000e+000 */ },
    { 0xfff0000000000000 /* -1.#INF00000000000e+000 */, 0xfff0000000000000 /* -1.#INF00000000000e+000 */ },
    { 0xfff0000000000001 /* -1.#QNAN0000000000e+000 */, 0xfff8000000000001 /* -1.#QNAN0000000000e+000 */ },
    { 0xfff7ffffffffffff /* -1.#QNAN0000000000e+000 */, 0xffffffffffffffff /* -1.#QNAN0000000000e+000 */ },
    { 0xfff8000000000000 /* -1.#IND00000000000e+000 */, 0xfff8000000000000 /* -1.#IND00000000000e+000 */ },
    { 0xfff8000000000001 /* -1.#QNAN0000000000e+000 */, 0xfff8000000000001 /* -1.#QNAN0000000000e+000 */ },
    { 0xffffffffffffffff /* -1.#QNAN0000000000e+000 */, 0xffffffffffffffff /* -1.#QNAN0000000000e+000 */ },

    /* Some random doubles */
    { 0x386580c747a3402b /*  5.055340589883462e-037 */, 0x0000000000000000 /*  0.000000000000000e+000 */ },
    { 0xb74298e6627fb9ed /* -1.667860443847725e-042 */, 0xbff0000000000000 /* -1.000000000000000e+000 */ },
    { 0x0ef25f06e414aa2d /*  1.128498317470960e-236 */, 0x0000000000000000 /*  0.000000000000000e+000 */ },
    { 0x24002a37167638b5 /*  2.780001692929186e-135 */, 0x0000000000000000 /*  0.000000000000000e+000 */ },
    { 0x44258d1be62a0d22 /*  1.987747995999515e+020 */, 0x44258d1be62a0d22 /*  1.987747995999515e+020 */ },
    { 0x9ed4e46a65aad464 /* -3.715074250469020e-160 */, 0xbff0000000000000 /* -1.000000000000000e+000 */ },
    { 0xc5afcd6f4ae4bf41 /* -4.921195330852160e+027 */, 0xc5afcd6f4ae4bf41 /* -4.921195330852160e+027 */ },
    { 0x330fac896cbb01d2 /*  9.624395081137827e-063 */, 0x0000000000000000 /*  0.000000000000000e+000 */ },
    { 0xc18026ab4c845405 /* -3.387120956461338e+007 */, 0xc18026ab50000000 /* -3.387121000000000e+007 */ },
    { 0x2f42a7dc898a741a /*  4.916804395045249e-081 */, 0x0000000000000000 /*  0.000000000000000e+000 */ },
};


void Test_floor(void)
{
    int i;

    for (i = 0; i < _countof(s_floor_tests); i++)
    {
        double x = u64_to_dbl(s_floor_tests[i].x);
        double z = floor(x);
        ok_eq_dbl_exact("floor", s_floor_tests[i].x, z, s_floor_tests[i].result);
    }
}

static TESTENTRY_FLT s_floorf_tests[] =
{
    /* Special values */
    { 0x00000000 /*  0.000000e+000 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x00000000 /*  0.000000e+000 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x00000000 /*  0.000000e+000 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x00000001 /*  1.401298e-045 */, 0x00000000 /*  0.000000e+000 */ },
    { 0xffffffff /* -1.#QNAN0e+000 */, 0xffffffff /* -1.#QNAN0e+000 */ },
    { 0x00000000 /*  0.000000e+000 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x00000001 /*  1.401298e-045 */, 0x00000000 /*  0.000000e+000 */ },
    { 0xffffffff /* -1.#QNAN0e+000 */, 0xffffffff /* -1.#QNAN0e+000 */ },
    { 0x00000000 /*  0.000000e+000 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x00000001 /*  1.401298e-045 */, 0x00000000 /*  0.000000e+000 */ },
    { 0xffffffff /* -1.#QNAN0e+000 */, 0xffffffff /* -1.#QNAN0e+000 */ },
    { 0x00000000 /*  0.000000e+000 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x00000001 /*  1.401298e-045 */, 0x00000000 /*  0.000000e+000 */ },
    { 0xffffffff /* -1.#QNAN0e+000 */, 0xffffffff /* -1.#QNAN0e+000 */ },

    /* Some random floats */
    { 0x386580c7 /*  5.471779e-005 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x47a3402b /*  8.358434e+004 */, 0x47a34000 /*  8.358400e+004 */ },
    { 0xb74298e6 /* -1.159890e-005 */, 0xbf800000 /* -1.000000e+000 */ },
    { 0x627fb9ed /*  1.179329e+021 */, 0x627fb9ed /*  1.179329e+021 */ },
    { 0x0ef25f06 /*  5.974911e-030 */, 0x00000000 /*  0.000000e+000 */ },
    { 0xe414aa2d /* -1.096952e+022 */, 0xe414aa2d /* -1.096952e+022 */ },
    { 0x24002a37 /*  2.779133e-017 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x167638b5 /*  1.988962e-025 */, 0x00000000 /*  0.000000e+000 */ },
    { 0x44258d1b /*  6.622048e+002 */, 0x44258000 /*  6.620000e+002 */ },
    { 0xe62a0d22 /* -2.007611e+023 */, 0xe62a0d22 /* -2.007611e+023 */ },
};

void Test_floorf(void)
{
    int i;

    for (i = 0; i < _countof(s_floorf_tests); i++)
    {
        float x = u32_to_flt(s_floorf_tests[i].x);
        float z = floorf(x);
        ok_eq_flt_exact("floorf", s_floorf_tests[i].x, z, s_floorf_tests[i].result);
    }
}

START_TEST(floor)
{
    Test_floor();
    Test_floorf();
}
