/*
 * Decompiled with CFR 0.152.
 */
package org.terasology.launcher;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import javafx.collections.ObservableSet;
import javafx.concurrent.Task;
import javafx.scene.control.Alert;
import javafx.scene.control.ButtonType;
import javafx.stage.Stage;
import org.kohsuke.github.GHRelease;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terasology.launcher.LauncherConfiguration;
import org.terasology.launcher.game.GameManager;
import org.terasology.launcher.model.GameIdentifier;
import org.terasology.launcher.model.GameRelease;
import org.terasology.launcher.model.LauncherVersion;
import org.terasology.launcher.repositories.RepositoryManager;
import org.terasology.launcher.settings.LauncherSettings;
import org.terasology.launcher.settings.LauncherSettingsValidator;
import org.terasology.launcher.settings.Settings;
import org.terasology.launcher.ui.Dialogs;
import org.terasology.launcher.updater.LauncherUpdater;
import org.terasology.launcher.util.BundleUtils;
import org.terasology.launcher.util.DirectoryCreator;
import org.terasology.launcher.util.FileUtils;
import org.terasology.launcher.util.HostServices;
import org.terasology.launcher.util.LauncherDirectoryUtils;
import org.terasology.launcher.util.LauncherManagedDirectory;
import org.terasology.launcher.util.LauncherStartFailedException;
import org.terasology.launcher.util.Platform;

public class LauncherInitTask
extends Task<LauncherConfiguration> {
    private static final Logger logger = LoggerFactory.getLogger(LauncherInitTask.class);
    private final Stage owner;
    private final HostServices hostServices;

    public LauncherInitTask(Stage newOwner, HostServices hostServices) {
        this.owner = newOwner;
        this.hostServices = hostServices;
    }

    protected LauncherConfiguration call() {
        try {
            Platform platform = this.getPlatform();
            this.updateMessage(BundleUtils.getLabel("splash_initLauncherDirs"));
            Path installationDirectory = LauncherDirectoryUtils.getInstallationDirectory();
            Path userDataDirectory = this.getLauncherDirectory(platform);
            Path downloadDirectory = this.getDirectoryFor(LauncherManagedDirectory.DOWNLOAD, userDataDirectory);
            Path tempDirectory = this.getDirectoryFor(LauncherManagedDirectory.TEMP, userDataDirectory);
            Path cacheDirectory = this.getDirectoryFor(LauncherManagedDirectory.CACHE, userDataDirectory);
            Path settingsFile = userDataDirectory.resolve("TerasologyLauncherSettings.properties");
            LauncherSettings launcherSettings = this.getLauncherSettings(settingsFile);
            LauncherSettingsValidator.validate(launcherSettings);
            this.checkForLauncherUpdates(downloadDirectory, tempDirectory, launcherSettings.isKeepDownloadedFiles());
            this.updateMessage(BundleUtils.getLabel("splash_initGameDirs"));
            Path gameDirectory = this.getDirectoryFor(LauncherManagedDirectory.GAMES, installationDirectory);
            Path gameDataDirectory = this.getGameDataDirectory(platform, launcherSettings.getGameDataDirectory());
            this.updateMessage(BundleUtils.getLabel("splash_fetchReleases"));
            logger.info("Fetching game releases ...");
            RepositoryManager repositoryManager = new RepositoryManager();
            Set<GameRelease> releases = repositoryManager.getReleases();
            GameManager gameManager = new GameManager(cacheDirectory, gameDirectory);
            ObservableSet<GameIdentifier> installedGames = gameManager.getInstalledGames();
            logger.trace("Change LauncherSettings...");
            launcherSettings.setGameDirectory(gameDirectory);
            launcherSettings.setGameDataDirectory(gameDataDirectory);
            this.storeLauncherSettingsAfterInit(launcherSettings, settingsFile);
            logger.trace("Creating launcher frame...");
            return new LauncherConfiguration(userDataDirectory, downloadDirectory, launcherSettings, gameManager, repositoryManager);
        }
        catch (LauncherStartFailedException e) {
            logger.warn("Could not configure launcher.");
            return null;
        }
    }

    private Platform getPlatform() {
        logger.trace("Init Platform...");
        this.updateMessage(BundleUtils.getLabel("splash_checkOS"));
        Platform platform = Platform.getPlatform();
        if (!(platform.isLinux() || platform.isMac() || platform.isWindows())) {
            logger.warn("Detected unexpected platform: {}", (Object)platform);
        }
        logger.debug("Platform: {}", (Object)platform);
        return platform;
    }

    private void initDirectory(Path dir, String errorLabel, DirectoryCreator ... creators) throws LauncherStartFailedException {
        try {
            for (DirectoryCreator creator : creators) {
                creator.apply(dir);
            }
        }
        catch (IOException e) {
            logger.error("Directory '{}' cannot be created or used! '{}'", dir.getFileName(), dir, e);
            Dialogs.showError(this.owner, BundleUtils.getLabel(errorLabel) + "\n" + dir);
            throw new LauncherStartFailedException();
        }
        logger.debug("{} directory: {}", (Object)dir.getFileName(), (Object)dir);
    }

    private Path getDirectoryFor(LauncherManagedDirectory directoryType, Path launcherDirectory) throws LauncherStartFailedException {
        Path dir = directoryType.getDirectoryPath(launcherDirectory);
        this.initDirectory(dir, directoryType.getErrorLabel(), directoryType.getCreators());
        return dir;
    }

    private Path getLauncherDirectory(Platform platform) throws LauncherStartFailedException {
        Path launcherDirectory = LauncherDirectoryUtils.getApplicationDirectory(platform, "TerasologyLauncher");
        this.initDirectory(launcherDirectory, "message_error_launcherDirectory", FileUtils::ensureWritableDir);
        return launcherDirectory;
    }

    private LauncherSettings getLauncherSettings(Path settingsFile) throws LauncherStartFailedException {
        logger.trace("Init LauncherSettings...");
        this.updateMessage(BundleUtils.getLabel("splash_retrieveLauncherSettings"));
        LauncherSettings settings = Optional.ofNullable(Settings.load(settingsFile)).orElse(Settings.getDefault());
        settings.init();
        logger.debug("Launcher Settings: {}", (Object)settings);
        return settings;
    }

    private void checkForLauncherUpdates(Path downloadDirectory, Path tempDirectory, boolean saveDownloadedFiles) {
        logger.trace("Check for launcher updates...");
        this.updateMessage(BundleUtils.getLabel("splash_launcherUpdateCheck"));
        LauncherUpdater updater = new LauncherUpdater(LauncherVersion.getInstance());
        GHRelease release = updater.updateAvailable();
        if (release != null) {
            boolean update;
            logger.info("Launcher update available: {}", (Object)release.getTagName());
            this.updateMessage(BundleUtils.getLabel("splash_launcherUpdateAvailable"));
            boolean foundLauncherInstallationDirectory = false;
            try {
                Path installationDir = LauncherDirectoryUtils.getInstallationDirectory();
                FileUtils.ensureWritableDir(installationDir);
                logger.trace("Launcher installation directory: {}", (Object)installationDir);
                foundLauncherInstallationDirectory = true;
            }
            catch (IOException e) {
                logger.error("The launcher installation directory can not be detected or used!", e);
                Dialogs.showError(this.owner, BundleUtils.getLabel("message_error_launcherInstallationDirectory"));
            }
            if (foundLauncherInstallationDirectory && (update = updater.showUpdateDialog(this.owner, release))) {
                this.showDownloadPage();
            }
        }
    }

    private void showDownloadPage() {
        String downloadPage = "https://terasology.org/download";
        try {
            this.hostServices.tryOpenUri(new URI("https://terasology.org/download"));
        }
        catch (URISyntaxException e) {
            logger.info("Could not open '{}': {}", (Object)"https://terasology.org/download", (Object)e.getMessage());
        }
    }

    private Path getGameDataDirectory(Platform os, Path settingsGameDataDirectory) throws LauncherStartFailedException {
        logger.trace("Init GameDataDirectory...");
        Path gameDataDirectory = settingsGameDataDirectory;
        if (gameDataDirectory != null) {
            try {
                FileUtils.ensureWritableDir(gameDataDirectory);
            }
            catch (IOException e) {
                logger.warn("The game data directory can not be created or used! '{}'", (Object)gameDataDirectory, (Object)e);
                Dialogs.showWarning(this.owner, BundleUtils.getLabel("message_error_gameDataDirectory") + "\n" + gameDataDirectory);
                gameDataDirectory = null;
            }
        }
        if (gameDataDirectory == null) {
            logger.trace("Choose data directory for the game...");
            this.updateMessage(BundleUtils.getLabel("splash_chooseGameDataDirectory"));
            gameDataDirectory = Dialogs.chooseDirectory(this.owner, LauncherDirectoryUtils.getGameDataDirectory(os), BundleUtils.getLabel("message_dialog_title_chooseGameDataDirectory"));
            if (Files.notExists(gameDataDirectory, new LinkOption[0])) {
                logger.info("The new game data directory is not approved. The TerasologyLauncher is terminated.");
                javafx.application.Platform.exit();
            }
        }
        try {
            FileUtils.ensureWritableDir(gameDataDirectory);
        }
        catch (IOException e) {
            logger.error("The game data directory can not be created or used! '{}'", (Object)gameDataDirectory, (Object)e);
            Dialogs.showError(this.owner, BundleUtils.getLabel("message_error_gameDataDirectory") + "\n" + gameDataDirectory);
            throw new LauncherStartFailedException();
        }
        logger.debug("Game data directory: {}", (Object)gameDataDirectory);
        return gameDataDirectory;
    }

    private boolean confirmSourcesOverwrite() {
        return CompletableFuture.supplyAsync(() -> {
            Alert alert = new Alert(Alert.AlertType.WARNING, BundleUtils.getLabel("message_error_sourcesFile_content"), new ButtonType[]{ButtonType.OK, new ButtonType(BundleUtils.getLabel("launcher_exit"))});
            alert.setHeaderText(BundleUtils.getLabel("message_error_sourcesFile_header"));
            alert.getDialogPane().setMinHeight(Double.NEGATIVE_INFINITY);
            return alert.showAndWait().map(btn -> btn == ButtonType.OK).orElse(false);
        }, javafx.application.Platform::runLater).join();
    }

    private void storeLauncherSettingsAfterInit(LauncherSettings launcherSettings, Path settingsFile) throws LauncherStartFailedException {
        logger.trace("Store LauncherSettings...");
        this.updateMessage(BundleUtils.getLabel("splash_storeLauncherSettings"));
        try {
            Settings.store(launcherSettings, settingsFile);
        }
        catch (IOException e) {
            logger.error("The launcher settings cannot be stored! '{}'", (Object)settingsFile, (Object)e);
            Dialogs.showError(this.owner, BundleUtils.getLabel("message_error_storeSettings"));
            throw new LauncherStartFailedException();
        }
        logger.debug("Launcher Settings stored: {}", (Object)launcherSettings);
    }
}

