/*
 * Decompiled with CFR 0.152.
 */
package org.terasology.launcher.repositories;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terasology.launcher.model.Build;
import org.terasology.launcher.model.GameIdentifier;
import org.terasology.launcher.model.GameRelease;
import org.terasology.launcher.model.Profile;
import org.terasology.launcher.model.ReleaseMetadata;
import org.terasology.launcher.repositories.Jenkins;
import org.terasology.launcher.repositories.JenkinsClient;
import org.terasology.launcher.repositories.ReleaseRepository;

class JenkinsRepositoryAdapter
implements ReleaseRepository {
    private static final Logger logger = LoggerFactory.getLogger(JenkinsRepositoryAdapter.class);
    private static final String BASE_URL = "http://jenkins.terasology.io/teraorg/job/Terasology/";
    private static final String API_FILTER = "api/json?tree=builds[number,timestamp,result,artifacts[fileName,relativePath],url]";
    private static final String TERASOLOGY_ZIP_PATTERN = "Terasology.*zip";
    private final JenkinsClient client;
    private final Build buildProfile;
    private final Profile profile;
    private final URL apiUrl;

    JenkinsRepositoryAdapter(Profile profile, Build buildProfile, JenkinsClient client) {
        this.client = client;
        this.buildProfile = buildProfile;
        this.profile = profile;
        this.apiUrl = JenkinsRepositoryAdapter.unsafeToUrl(BASE_URL + JenkinsRepositoryAdapter.job(JenkinsRepositoryAdapter.profileToJobName(profile)) + JenkinsRepositoryAdapter.job(JenkinsRepositoryAdapter.buildProfileToJobName(buildProfile)) + API_FILTER);
    }

    @Override
    public List<GameRelease> fetchReleases() {
        Jenkins.ApiResult result;
        LinkedList<GameRelease> pkgList = new LinkedList<GameRelease>();
        logger.debug("fetching releases from '{}'", (Object)this.apiUrl);
        try {
            result = this.client.request(this.apiUrl);
        }
        catch (InterruptedException e) {
            logger.warn("Interrupted while fetching packages from: {}", (Object)this.apiUrl, (Object)e);
            return Collections.emptyList();
        }
        if (result != null && result.builds != null) {
            for (Jenkins.Build build : result.builds) {
                this.computeReleaseFrom(build).ifPresent(pkgList::add);
            }
        } else {
            logger.warn("Failed to fetch packages from: {}", (Object)this.apiUrl);
        }
        return pkgList;
    }

    private Optional<GameRelease> computeReleaseFrom(Jenkins.Build jenkinsBuildInfo) {
        if (JenkinsRepositoryAdapter.hasAcceptableResult(jenkinsBuildInfo)) {
            URL url = this.client.getArtifactUrl(jenkinsBuildInfo, TERASOLOGY_ZIP_PATTERN);
            ReleaseMetadata metadata = this.computeReleaseMetadataFrom(jenkinsBuildInfo);
            Optional<GameIdentifier> id = this.computeIdentifierFrom(jenkinsBuildInfo);
            if (url != null && id.isPresent()) {
                return Optional.of(new GameRelease(id.get(), url, metadata));
            }
            logger.debug("Skipping build without game artifact or version identifier: '{}'", (Object)jenkinsBuildInfo.url);
        } else {
            logger.debug("Skipping unsuccessful build '{}'", (Object)jenkinsBuildInfo.url);
        }
        return Optional.empty();
    }

    private Optional<GameIdentifier> computeIdentifierFrom(Jenkins.Build jenkinsBuildInfo) {
        return Optional.ofNullable(this.client.getArtifactUrl(jenkinsBuildInfo, "versionInfo.properties")).map(this.client::requestProperties).map(versionInfo -> versionInfo.getProperty("displayVersion")).map(displayVersion -> {
            String versionString = displayVersion + "+" + jenkinsBuildInfo.number;
            return new GameIdentifier(versionString, this.buildProfile, this.profile);
        });
    }

    private ReleaseMetadata computeReleaseMetadataFrom(Jenkins.Build jenkinsBuildInfo) {
        String changelog = this.computeChangelogFrom(jenkinsBuildInfo.changeSet);
        Date timestamp = new Date(jenkinsBuildInfo.timestamp);
        return new ReleaseMetadata(changelog, timestamp);
    }

    private String computeChangelogFrom(Jenkins.ChangeSet changeSet) {
        return Optional.ofNullable(changeSet).map(changes -> Arrays.stream(changes.items).map(change -> "- " + change.msg).collect(Collectors.joining("\n"))).orElse("");
    }

    private static URL unsafeToUrl(String url) {
        try {
            return new URL(url);
        }
        catch (MalformedURLException malformedURLException) {
            return null;
        }
    }

    private static String profileToJobName(Profile profile) {
        switch (profile) {
            case OMEGA: {
                return "Omega/";
            }
            case ENGINE: {
                return "Terasology/";
            }
        }
        throw new IllegalStateException("Unexpected value: " + profile);
    }

    private static String buildProfileToJobName(Build buildProfile) {
        switch (buildProfile) {
            case STABLE: {
                return "master/";
            }
            case NIGHTLY: {
                return "develop/";
            }
        }
        throw new IllegalStateException("Unexpected value: " + buildProfile);
    }

    private static String job(String job) {
        return "job/" + job;
    }

    private static boolean hasAcceptableResult(Jenkins.Build build) {
        return build.result == Jenkins.Build.Result.SUCCESS || build.result == Jenkins.Build.Result.UNSTABLE;
    }
}

