/*
 * Decompiled with CFR 0.152.
 */
package org.terasology.launcher.game;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Collection;
import java.util.Comparator;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javafx.application.Platform;
import javafx.collections.FXCollections;
import javafx.collections.ObservableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terasology.launcher.game.Installation;
import org.terasology.launcher.model.Build;
import org.terasology.launcher.model.GameIdentifier;
import org.terasology.launcher.model.GameRelease;
import org.terasology.launcher.model.Profile;
import org.terasology.launcher.tasks.ProgressListener;
import org.terasology.launcher.util.DownloadException;
import org.terasology.launcher.util.DownloadUtils;
import org.terasology.launcher.util.FileUtils;

public class GameManager {
    private static final Logger logger = LoggerFactory.getLogger(GameManager.class);
    private final Path cacheDirectory;
    private final Path installDirectory;
    private final ObservableSet<GameIdentifier> installedGames;

    public GameManager(Path cacheDirectory, Path installDirectory) {
        this.cacheDirectory = cacheDirectory;
        this.installDirectory = installDirectory;
        this.installedGames = FXCollections.observableSet((Object[])new GameIdentifier[0]);
        this.scanInstallationDir();
    }

    private String getFileNameFor(GameRelease release) {
        GameIdentifier id = release.getId();
        String profileString = id.getProfile().toString().toLowerCase();
        String versionString = id.getDisplayVersion();
        String buildString = id.getBuild().toString().toLowerCase();
        return "terasology-" + profileString + "-" + versionString + "-" + buildString + ".zip";
    }

    public void install(GameRelease release, ProgressListener listener) throws IOException, DownloadException, InterruptedException {
        Path cachedZip = this.cacheDirectory.resolve(this.getFileNameFor(release));
        if (Files.notExists(cachedZip, new LinkOption[0])) {
            this.download(release, cachedZip, listener);
        }
        if (!listener.isCancelled()) {
            Path extractDir = this.getInstallDirectory(release.getId());
            FileUtils.extractZipTo(cachedZip, extractDir);
            Platform.runLater(() -> this.installedGames.add((Object)release.getId()));
            logger.info("Finished installing package: {}", (Object)release.getId());
        }
    }

    private void download(GameRelease release, Path targetLocation, ProgressListener listener) throws DownloadException, IOException, InterruptedException {
        URL downloadUrl = release.getUrl();
        long contentLength = DownloadUtils.getContentLength(downloadUrl);
        long availableSpace = targetLocation.getParent().toFile().getUsableSpace();
        if (availableSpace >= contentLength) {
            Path cacheZipPart = targetLocation.resolveSibling(targetLocation.getFileName().toString() + ".part");
            Files.deleteIfExists(cacheZipPart);
            try {
                DownloadUtils.downloadToFile(downloadUrl, cacheZipPart, listener).get();
            }
            catch (ExecutionException e) {
                throw new DownloadException("Exception while downloading " + downloadUrl, e.getCause());
            }
            if (!listener.isCancelled()) {
                Files.move(cacheZipPart, targetLocation, StandardCopyOption.ATOMIC_MOVE);
            }
        } else {
            throw new DownloadException("Insufficient space for downloading package");
        }
        logger.info("Finished downloading package: {}", (Object)release.getId());
    }

    public void remove(GameIdentifier game) throws IOException {
        Files.walk(this.getInstallDirectory(game), new FileVisitOption[0]).sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
        Platform.runLater(() -> this.installedGames.remove((Object)game));
        logger.info("Finished removing package: {}", (Object)game);
    }

    public ObservableSet<GameIdentifier> getInstalledGames() {
        return this.installedGames;
    }

    public Path getInstallDirectory(GameIdentifier id) {
        return this.installDirectory.resolve(id.getProfile().name()).resolve(id.getBuild().name()).resolve(id.getDisplayVersion());
    }

    public Installation getInstallation(GameIdentifier id) throws FileNotFoundException {
        return Installation.getExisting(this.getInstallDirectory(id));
    }

    private void scanInstallationDir() {
        Set localGames;
        try (Stream<Path> directories = Files.walk(this.installDirectory, 3, new FileVisitOption[0]);){
            Stream<Path> gameDirectories = directories.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).filter(d -> this.installDirectory.relativize((Path)d).getNameCount() == 3);
            localGames = gameDirectories.map(GameManager::getInstalledVersion).filter(Objects::nonNull).collect(Collectors.toUnmodifiableSet());
        }
        catch (IOException e) {
            logger.warn("Error while scanning installation directory {}:", (Object)this.installDirectory, (Object)e);
            return;
        }
        Platform.runLater(() -> this.installedGames.addAll((Collection)localGames));
    }

    private static GameIdentifier getInstalledVersion(Path versionDirectory) {
        Build build;
        Profile profile;
        int parts = versionDirectory.getNameCount();
        try {
            profile = Profile.valueOf(versionDirectory.getName(parts - 3).toString());
            build = Build.valueOf(versionDirectory.getName(parts - 2).toString());
        }
        catch (IllegalArgumentException e) {
            logger.debug("Directory does not match expected profile/build names: {}", (Object)versionDirectory, (Object)e);
            return null;
        }
        return new GameIdentifier(versionDirectory.getFileName().toString(), build, profile);
    }
}

