/*                         U N I T S . C
 * BRL-CAD
 *
 * Copyright (c) 1990-2021 United States Government as represented by
 * the U.S. Army Research Laboratory.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this file; see the file named COPYING for more
 * information.
 */

#include "common.h"

#ifdef HAVE_INTTYPES_H
#  if defined(__cplusplus)
#  define __STDC_FORMAT_MACROS
#  endif
#  include <inttypes.h>
#else
#  include "pinttypes.h"
#endif
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <float.h>
#include <limits.h>
#include "vmath.h"

#include "bu/log.h"
#include "bu/malloc.h"
#include "bu/str.h"
#include "bu/units.h"
#include "bu/vls.h"


struct cvt_tab {
    double val;
    char name[32];
};


struct conv_table {
    struct cvt_tab *cvttab;
};


/* Keep table sorted sorted small-to-large.  Algorithms below may rely
 * on the ordering.
 */
static struct cvt_tab bu_units_length_tab[] = {
    {0.0,		"none"},
    {1.0e-21,		"ym"},
    {1.0e-21,		"yoctometer"},
    {1.0e-18,		"zm"},
    {1.0e-18,		"zeptometer"},
    {1.0e-15,		"am"},
    {1.0e-15,		"attometer"},
    {1.0e-12,		"fm"},
    {1.0e-12,		"femtometer"},
    {1.0e-9,		"pm"},
    {1.0e-9,		"picometer"},
    {1.0e-7,		"angstrom"},
    {1.0e-7,		"decinanometer"},
    {1.0e-6,		"nm"},
    {1.0e-6,		"nanometer"},
    {1.0e-3,		"um"},
    {1.0e-3,		"micrometer"},
    {1.0e-3,		"micron"},
    {1.0,		"mm"},
    {1.0,		"millimeter"},
    {1.0,		"millimeters"}, /* plural */
    {10.0,		"cm"},
    {10.0,		"centimeter"},
    {10.0,		"centimeters"}, /* plural */
    {100.0,		"dm"},
    {100.0,		"decimeter"},
    {1000.0,		"m"},
    {1000.0,		"meter"},
    {1000.0,		"meters"}, /* plural */
    {10000.0,		"Dm"},
    {10000.0,		"decameter"},
    {100000.0,		"hm"},
    {100000.0,		"hectometer"},
    {1000000.0,		"km"},
    {1000000.0,		"kilometer"},
    {1.0e+9,		"Mm"},
    {1.0e+9,		"megameter"},
    {1.0e+12,		"Gm"},
    {1.0e+12,		"gigameter"},
    {1.0e+15,		"Tm"},
    {1.0e+15,		"terameter"},
    {1.0e+18,		"Pm"},
    {1.0e+18,		"petameter"},
    {1.0e+21,		"Em"},
    {1.0e+21,		"exameter"},
    {1.0e+24,		"Zm"},
    {1.0e+24,		"zettameter"},
    {1.0e+27,		"Ym"},
    {1.0e+27,		"yottameter"},
    {25.4,		"in"},
    {25.4,		"inch"},
    {25.4,		"inches"}, /* plural */
    {101.6,		"hand"},
    {304.8,		"ft"},
    {304.8,		"foot"},
    {304.8,		"feet"}, /* plural */
    {456.2,		"cubit"},
    {914.4,		"yd"},
    {914.4,		"yard"},
    {5029.2,		"rd"},
    {5029.2,		"rod"},
    {20116.8,		"chain"},
    {201168.0,		"furlong"},
    {1609344.0,		"mi"},
    {1609344.0,		"mile"},
    {1852000.0,		"nmile"},
    {1852000.0,		"nautical mile"},
    {5556000.0,		"league"},
    {2.99792458e+11,	"light second"},
    {1.79875475e+13,	"light minute"},
    {1.495979e+14,	"AU"},
    {1.495979e+14,	"astronomical unit"},
    {1.07925285e+15,	"light hour"},
    {2.59020684e+16,	"light day"},
    {9.4605284e+18,	"light year"},
    {3.08568025e+19,	"pc"},
    {3.08568025e+19,	"parsec"},
    {0.0,		""}			/* LAST ENTRY */
};
#define BU_UNITS_TABLE_SIZE (sizeof(bu_units_length_tab) / sizeof(struct cvt_tab) - 1)

/* Keep table sorted sorted small-to-large.  Algorithms below may rely
 * on the ordering.
 */
static struct cvt_tab bu_units_volume_tab[] = {
    {0.0,		"none"},
    {1.0,		"mm^3"},		/* default */
    {1.0, 		"cu mm"},
    {1.0e3, 		"cm^3"},
    {1.0e3, 		"cu cm"},
    {1.0e3, 		"cc"},
    {1.0e6, 		"l"},
    {1.0e6, 		"liter"},
    {1.0e6, 		"litre"},
    {1.0e9, 		"m^3"},
    {1.0e9,		"cu m"},
    {16387.064, 	"in^3"},
    {16387.064, 	"cu in"},
    {28316846.592, 	"ft^3"},
    {28316846.592, 	"cu ft"},
    {764554857.984, 	"yds^3"},
    {764554857.984, 	"yards^3"},
    {764554857.984, 	"cu yards"},
    {0.0,               ""}                     /* LAST ENTRY */
};


/* Keep table sorted sorted small-to-large.  Algorithms below may rely
 * on the ordering.
 */
static struct cvt_tab bu_units_mass_tab[] = {
    {0.0,		"none"},
    {1.0,		"grams"},		/* default */
    {1.0, 		"g"},
    {1.0e3, 		"kilogram"},
    {1.0e3,		"kg"},
    {0.0648, 		"gr"},
    {0.0648,		"grain"},
    {453.6,		"lb"},
    {28.35,		"oz"},
    {28.35,		"ounce"},
    {0.0,               ""}                     /* LAST ENTRY */
};


static const struct conv_table unit_lists[4] = {
    {bu_units_length_tab}, {bu_units_volume_tab}, {bu_units_mass_tab}, {NULL}
};


/**
 * compares an input units string to a reference units name and
 * returns truthfully if they match.  the comparison ignores any
 * embedded whitespace and is case insensitive.
 */
static int
units_name_matches(const char *input, const char *name)
{
    const char *cp;
    int match;
    struct bu_vls normalized_input = BU_VLS_INIT_ZERO;
    struct bu_vls normalized_name = BU_VLS_INIT_ZERO;

    /* convert NULL */
    if (!input)
	input = "";
    if (!name)
	name = "";

    /* skip spaces */
    while (isspace((unsigned char)*input))
	input++;
    while (isspace((unsigned char)*name))
	name++;

    /* quick exit */
    if (tolower((unsigned char)input[0]) != tolower((unsigned char)name[0]))
	return 0;

    cp = input;
    /* skip spaces, convert to lowercase */
    while (*cp != '\0') {
	if (!isspace((unsigned char)*cp))
	    bu_vls_putc(&normalized_input, tolower((unsigned char)*cp));
	cp++;
    }

    cp = name;
    /* skip spaces, convert to lowercase */
    while (*cp != '\0') {
	if (!isspace((unsigned char)*cp))
	    bu_vls_putc(&normalized_name, tolower((unsigned char)*cp));
	cp++;
    }

    /* trim any trailing 's' for plurality */
    if (bu_vls_addr(&normalized_input)[bu_vls_strlen(&normalized_input)-1] == 's') {
	bu_vls_trunc(&normalized_input, -1);
    }
    if (bu_vls_addr(&normalized_name)[bu_vls_strlen(&normalized_name)-1] == 's') {
	bu_vls_trunc(&normalized_name, -1);
    }

    /* compare */
    match = BU_STR_EQUAL(bu_vls_addr(&normalized_input), bu_vls_addr(&normalized_name));

    bu_vls_free(&normalized_input);
    bu_vls_free(&normalized_name);

    return match;
}


double
bu_units_conversion(const char *str)
{
    const struct cvt_tab *tp;
    const struct conv_table *cvtab;
    double factor = 1.0;

    /* Search for the units string in the table matching by name. */
    for (cvtab=unit_lists; cvtab->cvttab; cvtab++) {
	for (tp=cvtab->cvttab; tp->name[0]; tp++) {
	    if (units_name_matches(str, tp->name)) {
		return tp->val;
	    }
	}
    }
    /* couldn't find it by name, check by value */
    factor = bu_mm_value(str);
    if (!EQUAL(factor, 1.0)) {
	for (cvtab=unit_lists; cvtab->cvttab; cvtab++) {
	    for (tp=cvtab->cvttab; tp->name[0]; tp++) {
		if (EQUAL(factor, tp->val)) {
		    return tp->val;
		}
	    }
	}
    }
    return 0.0;		/* Unable to find it */
}


const char *
bu_units_string(register const double mm)
{
    register const struct cvt_tab *tp;

    if (UNLIKELY(mm <= 0))
	return (const char *)NULL;

    /* Search for this string in the table */
    for (tp=bu_units_length_tab; tp->name[0]; tp++) {
	fastf_t diff, bigger;

	if (ZERO(mm - tp->val))
	    return tp->name;

	/* Check for near-miss */
	if (mm > tp->val) {
	    bigger = mm;
	    diff = mm - tp->val;
	} else {
	    bigger = tp->val;
	    diff = tp->val - mm;
	}

	/* Absolute difference less than 0.1 angstrom */
	if (diff < 1.0e-8)
	    return tp->name;

	/* Relative difference less than 1 part per billion */
	if (diff < 0.000000001 * bigger)
	    return tp->name;
    }
    return (const char *)NULL;
}


struct bu_vls *
bu_units_strings_vls()
{
    register const struct cvt_tab *tp;
    struct bu_vls *vlsp;
    double prev_val = 0.0;

    BU_ALLOC(vlsp, struct bu_vls);
    bu_vls_init(vlsp);
    for (tp=bu_units_length_tab; tp->name[0]; tp++) {
	if (ZERO(prev_val - tp->val))
	    continue;

	bu_vls_printf(vlsp, "%s, ", tp->name);
	prev_val = tp->val;
    }

    /* Remove the last ", " */
    bu_vls_trunc(vlsp, -2);

    return vlsp;
}


const char *
bu_nearest_units_string(register const double mm)
{
    register const struct cvt_tab *tp;

    const char *nearest = NULL;
    double nearer = DBL_MAX;

    if (UNLIKELY(mm <= 0))
	return (const char *)NULL;

    /* Search for this unit in the table */
    for (tp=bu_units_length_tab; tp->name[0]; tp++) {
	double nearness;

	/* skip zero so we don't return 'none' */
	if (ZERO(tp->val))
	    continue;

	/* break early on perfect match */
	if (ZERO(mm - tp->val))
	    return tp->name;

	/* Check for nearness */
	if (mm > tp->val) {
	    nearness = mm - tp->val;
	} else {
	    nearness = tp->val - mm;
	}

	/* :-) */
	if (nearness < nearer) {
	    nearer = nearness;
	    nearest = tp->name;
	}
    }
    return nearest;
}


double
bu_mm_value(const char *s)


{
    double v;
    char *ptr;
    register const struct cvt_tab *tp;

    v = strtod(s, &ptr);

    if (ptr == s) {
	/* No number could be found, unity is implied */
	/* e.g. interpret "ft" as "1ft" */
	v = 1.0;
    }
    if (! *ptr) {
	/* There are no characters following the scanned number */
	return v;
    }

    for (tp=bu_units_length_tab; tp->name[0]; tp++) {
	if (units_name_matches(ptr, tp->name)) {
	    v *= tp->val;
	    return v;
	}
    }

    /* A string was seen, but not found in the table.  Signal error */
    return -1.0;
}


void
bu_mm_cvt(const struct bu_structparse *sdp,
	  const char *name,
	  void *base,
	  const char *value,
	  void *UNUSED(data))
/* structure description */
/* struct member name */
/* beginning of structure */
/* string containing value */
{
    register double *p = (double *)((char *)base + sdp->sp_offset);

    if (UNLIKELY(!name)) {
	bu_log("bu_mm_cvt: NULL name encountered\n");
    }

    /* reconvert with optional units */
    *p = bu_mm_value(value);
}


/*
 * Local Variables:
 * mode: C
 * tab-width: 8
 * indent-tabs-mode: t
 * c-file-style: "stroustrup"
 * End:
 * ex: shiftwidth=4 tabstop=8
 */
