/*                   N U R B _ X S P L I T . C
 * BRL-CAD
 *
 * Copyright (c) 1990-2021 United States Government as represented by
 * the U.S. Army Research Laboratory.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this file; see the file named COPYING for more
 * information.
 */
/** @addtogroup nurb */
/** @{ */
/** @file primitives/bspline/nurb_xsplit.c
 *
 * Subdivide a nurb surface by inserting a multiple knot of of the
 * surface order in a given direction and return the resulting
 * surfaces.
 *
 */
/** @} */

#include "common.h"

#include "bio.h"

#include "vmath.h"
#include "bu/malloc.h"
#include "nmg.h"

/**
 * Algorithm
 *
 * Given a parametric direction (u or v) look at the direction knot
 * vector and insert a multiple knot of parametric direction surface
 * order. This is somewhat different than rt_nurb_split in that the
 * surface is give a parametric value at which to split the surface.
 * nmg_nurb_kvmult does the right thing in inserting a multiple knot
 * with the correct amount. Separate the surface and return the two
 * resulting surface.
 */
struct face_g_snurb *
nmg_nurb_s_xsplit(struct face_g_snurb *srf, fastf_t param, int dir)
{
    struct knot_vector new_kv;
    struct oslo_mat * oslo;
    int i;
    int k_index;
    struct face_g_snurb * srf1, * srf2;

    NMG_CK_SNURB(srf);

    if (dir == RT_NURB_SPLIT_ROW) {
	nmg_nurb_kvmult(&new_kv, &srf->u, srf->order[0], param);

	k_index = srf->order[0];

	oslo = (struct oslo_mat *)
	    nmg_nurb_calc_oslo(srf->order[RT_NURB_SPLIT_ROW], &srf->u, &new_kv);

	GET_SNURB(srf1);
	srf1->order[0]  = srf->order[0];
	srf1->order[1]  = srf->order[1];
	srf1->dir = RT_NURB_SPLIT_ROW;
	nmg_nurb_kvextract(&srf1->u, &new_kv, 0, k_index + srf1->order[0]);
	nmg_nurb_kvcopy(&srf1->v, &srf->v);

	srf1->pt_type = srf->pt_type;
	srf1->s_size[0] = srf1->v.k_size -
	    srf1->order[1];
	srf1->s_size[1] = srf1->u.k_size -
	    srf1->order[0];

	srf1->ctl_points = (fastf_t *)
	    nmg_malloc(sizeof(fastf_t) * srf1->s_size[0] *
		      srf1->s_size[1] *
		      RT_NURB_EXTRACT_COORDS(srf1->pt_type),
		      "nmg_nurb_s_xsplit: srf1 row mesh control points");

	GET_SNURB(srf2);
	srf2->order[0]  = srf->order[0];
	srf2->order[1]  = srf->order[1];
	srf2->dir = RT_NURB_SPLIT_ROW;
	nmg_nurb_kvextract(&srf2->u, &new_kv, k_index, new_kv.k_size);
	nmg_nurb_kvcopy(&srf2->v, &srf->v);

	srf2->pt_type = srf->pt_type;
	srf2->s_size[0] = srf2->v.k_size -
	    srf2->order[1];
	srf2->s_size[1] = srf2->u.k_size -
	    srf2->order[0];

	srf2->ctl_points = (fastf_t *)
	    nmg_malloc(sizeof(fastf_t) * srf2->s_size[0] *
		      srf2->s_size[1] *
		      RT_NURB_EXTRACT_COORDS(srf2->pt_type),
		      "nmg_nurb_s_xsplit: srf2 row mesh control points");

	for (i = 0; i < srf->s_size[0]; i++) {
	    fastf_t * old_mesh_ptr;
	    fastf_t * new_mesh_ptr;

	    old_mesh_ptr = &srf->ctl_points[
		i * srf->s_size[1] *
		RT_NURB_EXTRACT_COORDS(srf->pt_type)];
	    new_mesh_ptr = &srf1->ctl_points[
		i * srf1->s_size[1] *
		RT_NURB_EXTRACT_COORDS(srf1->pt_type)];
	    nmg_nurb_map_oslo(oslo, old_mesh_ptr, new_mesh_ptr,
			     RT_NURB_EXTRACT_COORDS(srf->pt_type),
			     RT_NURB_EXTRACT_COORDS(srf1->pt_type),
			     0, k_index, srf1->pt_type);
	    new_mesh_ptr = &srf2->ctl_points[
		i * srf2->s_size[1] *
		RT_NURB_EXTRACT_COORDS(srf2->pt_type)];
	    nmg_nurb_map_oslo(oslo, old_mesh_ptr, new_mesh_ptr,
			     RT_NURB_EXTRACT_COORDS(srf->pt_type),
			     RT_NURB_EXTRACT_COORDS(srf2->pt_type),
			     k_index, new_kv.k_size - srf2->order[0],
			     srf2->pt_type);
	}
    } else {
	nmg_nurb_kvmult(&new_kv, &srf->v, srf->order[RT_NURB_SPLIT_COL], param);

	k_index = srf->order[1];

	oslo = (struct oslo_mat *)
	    nmg_nurb_calc_oslo(srf->order[RT_NURB_SPLIT_COL], &srf->v, &new_kv);

	GET_SNURB(srf1);
	srf1->order[0]  = srf->order[0];
	srf1->order[1]  = srf->order[1];
	srf1->dir = RT_NURB_SPLIT_COL;
	nmg_nurb_kvextract(&srf1->v, &new_kv, 0, k_index + srf1->order[RT_NURB_SPLIT_COL]);
	nmg_nurb_kvcopy(&srf1->u, &srf->u);

	srf1->pt_type = srf->pt_type;
	srf1->s_size[0] = srf1->v.k_size -
	    srf1->order[1];
	srf1->s_size[1] = srf1->u.k_size -
	    srf1->order[0];

	srf1->ctl_points = (fastf_t *)
	    nmg_malloc(sizeof(fastf_t) * srf1->s_size[0] *
		      srf1->s_size[1] *
		      RT_NURB_EXTRACT_COORDS(srf1->pt_type),
		      "rt_nurb_split: srf1 row mesh control points");

	GET_SNURB(srf2);
	srf2->order[0]  = srf->order[0];
	srf2->order[1]  = srf->order[1];
	srf2->dir = RT_NURB_SPLIT_COL;
	nmg_nurb_kvextract(&srf2->v, &new_kv, k_index, new_kv.k_size);
	nmg_nurb_kvcopy(&srf2->u, &srf->u);

	srf2->pt_type = srf->pt_type;
	srf2->s_size[0] = srf2->v.k_size -
	    srf2->order[1];
	srf2->s_size[1] = srf2->u.k_size -
	    srf2->order[0];

	srf2->ctl_points = (fastf_t *)
	    nmg_malloc(sizeof(fastf_t) * srf2->s_size[0] *
		      srf2->s_size[1] *
		      RT_NURB_EXTRACT_COORDS(srf2->pt_type),
		      "nmg_nurb_s_xsplit: srf2 row mesh control points");

	for (i = 0; i < srf->s_size[1]; i++) {
	    fastf_t * old_mesh_ptr;
	    fastf_t * new_mesh_ptr;

	    old_mesh_ptr = &srf->ctl_points[
		i * RT_NURB_EXTRACT_COORDS(srf->pt_type)];
	    new_mesh_ptr = &srf1->ctl_points[
		i * RT_NURB_EXTRACT_COORDS(srf1->pt_type)];
	    nmg_nurb_map_oslo(oslo, old_mesh_ptr, new_mesh_ptr,
			     srf->s_size[1] *
			     RT_NURB_EXTRACT_COORDS(srf->pt_type),
			     srf1->s_size[1] *
			     RT_NURB_EXTRACT_COORDS(srf1->pt_type),
			     0, k_index, srf1->pt_type);
	    new_mesh_ptr = &srf2->ctl_points[
		i * RT_NURB_EXTRACT_COORDS(srf2->pt_type)];
	    nmg_nurb_map_oslo(oslo, old_mesh_ptr, new_mesh_ptr,
			     srf->s_size[1] *
			     RT_NURB_EXTRACT_COORDS(srf->pt_type),
			     srf2->s_size[1] *
			     RT_NURB_EXTRACT_COORDS(srf2->pt_type),
			     k_index, new_kv.k_size - srf2->order[1],
			     srf2->pt_type);
	}
    }

    BU_LIST_APPEND(&srf1->l, &srf2->l);

    nmg_free((char *) new_kv.knots, "nmg_nurb_s_xsplit: new_kv.knots");

    nmg_nurb_free_oslo(oslo);

    return srf1;
}


/**
 * nmg_nurb_c_xsplit()
 *
 * Split a NURB curve by inserting a multiple knot and return the
 * result of the two curves.
 *
 * Algorithm:
 *
 * Insert a multiple knot of the curve order. A parameter is give for
 * the knot value for which the curve will be split.
 */
struct edge_g_cnurb *
nmg_nurb_c_xsplit(struct edge_g_cnurb *crv, fastf_t param)
{
    struct knot_vector new_kv;
    struct oslo_mat * oslo;
    int k_index;
    struct edge_g_cnurb * crv1, * crv2;
    int coords;

    NMG_CK_CNURB(crv);

    coords = RT_NURB_EXTRACT_COORDS(crv->pt_type);

	k_index = crv->order;
    nmg_nurb_kvmult(&new_kv, &crv->k, crv->order, param);

    oslo = (struct oslo_mat *)
	nmg_nurb_calc_oslo(crv->order, &crv->k, &new_kv);

    GET_CNURB(crv1);
    crv1->order  = crv->order;
    nmg_nurb_kvextract(&crv1->k, &new_kv, 0, k_index + crv->order);
    crv1->pt_type = crv->pt_type;
    crv1->c_size = crv1->k.k_size - crv1->order;
    crv1->ctl_points = (fastf_t *)
	nmg_malloc(sizeof(fastf_t) * crv1->c_size *
		  RT_NURB_EXTRACT_COORDS(crv1->pt_type),
		  "nmg_nurb_c_xsplit: crv1 control points");

    GET_CNURB(crv2);
    crv2->order  = crv->order;
    nmg_nurb_kvextract(&crv2->k, &new_kv, k_index, new_kv.k_size);
    crv2->pt_type = crv->pt_type;
    crv2->c_size = crv2->k.k_size - crv2->order;
    crv2->ctl_points = (fastf_t *)
	nmg_malloc(sizeof(fastf_t) * crv2->c_size *
		  RT_NURB_EXTRACT_COORDS(crv2->pt_type),
		  "nmg_nurb_c_xsplit: crv2 row mesh control points");

    nmg_nurb_map_oslo(oslo, crv->ctl_points, crv1->ctl_points,
		     coords, coords, 0, k_index, crv->pt_type);

    nmg_nurb_map_oslo(oslo, crv->ctl_points, crv2->ctl_points,
		     coords, coords, k_index, new_kv.k_size - crv2->order,
		     crv2->pt_type);

    nmg_nurb_free_oslo(oslo);

    nmg_free((char *) new_kv.knots, "nmg_nurb_c_xsplit: new_kv.knots");

    BU_LIST_APPEND(&crv1->l, &crv2->l);
    return crv1;
}


/*
 * Local Variables:
 * mode: C
 * tab-width: 8
 * indent-tabs-mode: t
 * c-file-style: "stroustrup"
 * End:
 * ex: shiftwidth=4 tabstop=8
 */
