/*                        E P S P S I . C
 * BRL-CAD
 *
 * Published in 2020 by the United States Government.
 * This work is in the public domain.
 *
 */
/** @file epspsi.c
 *
 * BRL-CAD's translation of the calculation of coordinates for NACA airfoils is
 * based on the public domain program naca456 written by Ralph Carmichael of
 * Public Domain Aeronautical Software (PDAS):
 *
 * http://www.pdas.com/naca456.html
 *
 * naca456 is in turn based off of earlier work by several authors at NASA,
 * documented in reports NASA TM X-3284, NASA TM X-3069 and NASA TM 4741. The
 * program naca456 is documented in the paper:
 *
 * Carmichael, Ralph L.: Algorithm for Calculating Coordinates of Cambered NACA
 * Airfoils At Specified Chord Locations. AIAA Paper 2001-5235, November 2001.
 *
 * Disclaimer, per the PDAS distribution:
 *
 * Although many of the works contained herein were developed by national
 * laboratories or their contractors, neither the U.S. Government nor Public
 * Domain Aeronautical Software make any warranty as to the accuracy or
 * appropriateness of the procedure to any particular application.
 *
 * The programs and descriptions have been collected and reproduced with great
 * care and attention to accuracy, but no guarantee or warranty is implied. All
 * programs are offered AS IS and Public Domain Aeronautical Software does not
 * give any express or implied warranty of any kind and any implied warranties are
 * disclaimed. Public Domain Aeronautical Software will not be liable for any
 * direct, indirect, special, incidental, or consequential damages arising out of
 * any use of this software. In no case should the results of any computational
 * scheme be used as a substitute for sound engineering practice and judgment.
 *
 */

#include "naca.h"

const fastf_t EPS1[] = {
    0.00000, 0.00164, 0.00327, 0.00487, 0.00641, 0.00790, 0.00928, 0.01057,
    0.01174, 0.01278, 0.01367, 0.01439, 0.01497, 0.01542, 0.01576, 0.01601,
    0.01619, 0.01632, 0.01642, 0.01651, 0.01661, 0.01673, 0.01688, 0.01705,
    0.01725, 0.01747, 0.01771, 0.01797, 0.01824, 0.01853, 0.01884, 0.01916,
    0.01949, 0.01984, 0.02020, 0.02058, 0.02097, 0.02137, 0.02179, 0.02223,
    0.02268, 0.02315, 0.02363, 0.02413, 0.02464, 0.02517, 0.02571, 0.02626,
    0.02683, 0.02741, 0.02801, 0.02862, 0.02924, 0.02988, 0.03052, 0.03118,
    0.03185, 0.03253, 0.03323, 0.03393, 0.03465, 0.03538, 0.03611, 0.03686,
    0.03762, 0.03839, 0.03917, 0.03995, 0.04075, 0.04156, 0.04237, 0.04319,
    0.04402, 0.04486, 0.04571, 0.04657, 0.04743, 0.04831, 0.04919, 0.05008,
    0.05098, 0.05189, 0.05280, 0.05372, 0.05464, 0.05556, 0.05648, 0.05740,
    0.05831, 0.05921, 0.06011, 0.06099, 0.06187, 0.06273, 0.06357, 0.06440,
    0.06522, 0.06602, 0.06681, 0.06757, 0.06832, 0.06905, 0.06976, 0.07044,
    0.07111, 0.07176, 0.07238, 0.07298, 0.07356, 0.07411, 0.07464, 0.07514,
    0.07562, 0.07607, 0.07650, 0.07690, 0.07727, 0.07761, 0.07793, 0.07822,
    0.07848, 0.07871, 0.07891, 0.07908, 0.07922, 0.07933, 0.07941, 0.07945,
    0.07946, 0.07944, 0.07938, 0.07929, 0.07916, 0.07900, 0.07880, 0.07856,
    0.07829, 0.07799, 0.07764, 0.07726, 0.07685, 0.07640, 0.07591, 0.07539,
    0.07483, 0.07423, 0.07359, 0.07293, 0.07222, 0.07148, 0.07070, 0.06989,
    0.06904, 0.06815, 0.06723, 0.06628, 0.06529, 0.06427, 0.06322, 0.06214,
    0.06103, 0.05989, 0.05871, 0.05751, 0.05628, 0.05502, 0.05374, 0.05243,
    0.05109, 0.04973, 0.04834, 0.04693, 0.04549, 0.04404, 0.04256, 0.04106,
    0.03955, 0.03802, 0.03647, 0.03491, 0.03333, 0.03174, 0.03014, 0.02853,
    0.02690, 0.02527, 0.02363, 0.02198, 0.02032, 0.01865, 0.01698, 0.01530,
    0.01361, 0.01192, 0.01023, 0.00853, 0.00683, 0.00512, 0.00342, 0.00171,
    0.00000
};

const fastf_t EPS2[] = {
    0.00000, 0.00233, 0.00465, 0.00693, 0.00915, 0.01130, 0.01337, 0.01532,
    0.01715, 0.01884, 0.02035, 0.02169, 0.02288, 0.02391, 0.02481, 0.02557,
    0.02624, 0.02682, 0.02731, 0.02774, 0.02812, 0.02846, 0.02877, 0.02905,
    0.02931, 0.02957, 0.02982, 0.03007, 0.03033, 0.03060, 0.03090, 0.03122,
    0.03158, 0.03196, 0.03236, 0.03280, 0.03326, 0.03375, 0.03427, 0.03481,
    0.03538, 0.03598, 0.03660, 0.03725, 0.03792, 0.03862, 0.03935, 0.04010,
    0.04087, 0.04167, 0.04250, 0.04335, 0.04423, 0.04512, 0.04605, 0.04699,
    0.04796, 0.04896, 0.04998, 0.05102, 0.05208, 0.05317, 0.05428, 0.05541,
    0.05657, 0.05774, 0.05894, 0.06016, 0.06140, 0.06267, 0.06395, 0.06526,
    0.06658, 0.06793, 0.06931, 0.07070, 0.07213, 0.07357, 0.07505, 0.07655,
    0.07808, 0.07964, 0.08123, 0.08284, 0.08447, 0.08613, 0.08780, 0.08949,
    0.09119, 0.09290, 0.09462, 0.09635, 0.09808, 0.09980, 0.10151, 0.10321,
    0.10488, 0.10653, 0.10815, 0.10972, 0.11125, 0.11273, 0.11415, 0.11553,
    0.11686, 0.11814, 0.11938, 0.12057, 0.12171, 0.12281, 0.12386, 0.12487,
    0.12583, 0.12675, 0.12762, 0.12844, 0.12922, 0.12994, 0.13062, 0.13125,
    0.13182, 0.13234, 0.13281, 0.13322, 0.13358, 0.13389, 0.13414, 0.13434,
    0.13448, 0.13456, 0.13459, 0.13456, 0.13447, 0.13433, 0.13413, 0.13387,
    0.13354, 0.13316, 0.13272, 0.13222, 0.13166, 0.13104, 0.13035, 0.12960,
    0.12879, 0.12792, 0.12698, 0.12598, 0.12492, 0.12380, 0.12261, 0.12136,
    0.12004, 0.11866, 0.11723, 0.11573, 0.11417, 0.11255, 0.11087, 0.10914,
    0.10735, 0.10550, 0.10361, 0.10165, 0.09964, 0.09758, 0.09546, 0.09329,
    0.09106, 0.08878, 0.08645, 0.08406, 0.08163, 0.07914, 0.07661, 0.07403,
    0.07140, 0.06874, 0.06604, 0.06329, 0.06052, 0.05770, 0.05487, 0.05199,
    0.04908, 0.04615, 0.04319, 0.04021, 0.03721, 0.03417, 0.03113, 0.02807,
    0.02499, 0.02189, 0.01879, 0.01567, 0.01255, 0.00942, 0.00628, 0.00314,
    0.00000
};

const fastf_t EPS3[] = {
    0.00000, 0.00165, 0.00330, 0.00493, 0.00653, 0.00810, 0.00963, 0.01110,
    0.01253, 0.01387, 0.01515, 0.01625, 0.01702, 0.01746, 0.01771, 0.01795,
    0.01824, 0.01858, 0.01893, 0.01922, 0.01943, 0.01951, 0.01945, 0.01926,
    0.01899, 0.01865, 0.01830, 0.01794, 0.01760, 0.01733, 0.01715, 0.01704,
    0.01702, 0.01706, 0.01715, 0.01727, 0.01742, 0.01759, 0.01777, 0.01798,
    0.01821, 0.01846, 0.01876, 0.01908, 0.01944, 0.01983, 0.02024, 0.02068,
    0.02114, 0.02161, 0.02211, 0.02261, 0.02313, 0.02365, 0.02419, 0.02474,
    0.02530, 0.02587, 0.02647, 0.02708, 0.02772, 0.02837, 0.02904, 0.02974,
    0.03046, 0.03119, 0.03195, 0.03271, 0.03350, 0.03429, 0.03510, 0.03593,
    0.03677, 0.03763, 0.03849, 0.03938, 0.04028, 0.04119, 0.04212, 0.04307,
    0.04404, 0.04502, 0.04602, 0.04704, 0.04808, 0.04914, 0.05021, 0.05130,
    0.05241, 0.05353, 0.05467, 0.05582, 0.05699, 0.05817, 0.05935, 0.06055,
    0.06174, 0.06294, 0.06414, 0.06534, 0.06653, 0.06772, 0.06890, 0.07006,
    0.07121, 0.07234, 0.07346, 0.07456, 0.07564, 0.07668, 0.07771, 0.07872,
    0.07970, 0.08064, 0.08155, 0.08244, 0.08327, 0.08406, 0.08480, 0.08550,
    0.08614, 0.08673, 0.08728, 0.08778, 0.08825, 0.08866, 0.08905, 0.08939,
    0.08968, 0.08995, 0.09017, 0.09034, 0.09048, 0.09057, 0.09062, 0.09061,
    0.09057, 0.09046, 0.09031, 0.09009, 0.08982, 0.08950, 0.08911, 0.08868,
    0.08819, 0.08765, 0.08707, 0.08643, 0.08576, 0.08504, 0.08427, 0.08346,
    0.08260, 0.08170, 0.08075, 0.07976, 0.07871, 0.07759, 0.07640, 0.07510,
    0.07368, 0.07216, 0.07047, 0.06864, 0.06664, 0.06448, 0.06220, 0.05982,
    0.05735, 0.05483, 0.05228, 0.04972, 0.04718, 0.04467, 0.04225, 0.03991,
    0.03766, 0.03548, 0.03338, 0.03135, 0.02939, 0.02752, 0.02569, 0.02394,
    0.02224, 0.02059, 0.01899, 0.01744, 0.01592, 0.01446, 0.01302, 0.01163,
    0.01026, 0.00892, 0.00761, 0.00630, 0.00502, 0.00376, 0.00250, 0.00124,
    0.00000
};

const fastf_t EPS4[] = {
    0.00000, 0.00145, 0.00290, 0.00433, 0.00574, 0.00712, 0.00847, 0.00978,
    0.01105, 0.01225, 0.01340, 0.01447, 0.01547, 0.01638, 0.01719, 0.01789,
    0.01847, 0.01893, 0.01924, 0.01940, 0.01940, 0.01924, 0.01893, 0.01850,
    0.01799, 0.01741, 0.01679, 0.01616, 0.01556, 0.01499, 0.01450, 0.01410,
    0.01379, 0.01356, 0.01340, 0.01331, 0.01327, 0.01328, 0.01333, 0.01340,
    0.01350, 0.01361, 0.01373, 0.01387, 0.01402, 0.01419, 0.01438, 0.01458,
    0.01480, 0.01504, 0.01530, 0.01558, 0.01588, 0.01620, 0.01654, 0.01689,
    0.01726, 0.01765, 0.01805, 0.01847, 0.01890, 0.01934, 0.01980, 0.02026,
    0.02074, 0.02124, 0.02174, 0.02226, 0.02279, 0.02334, 0.02390, 0.02447,
    0.02506, 0.02566, 0.02627, 0.02690, 0.02754, 0.02819, 0.02885, 0.02952,
    0.03020, 0.03089, 0.03160, 0.03231, 0.03304, 0.03378, 0.03453, 0.03530,
    0.03608, 0.03688, 0.03770, 0.03853, 0.03938, 0.04025, 0.04113, 0.04202,
    0.04293, 0.04386, 0.04479, 0.04574, 0.04670, 0.04767, 0.04866, 0.04966,
    0.05067, 0.05171, 0.05277, 0.05386, 0.05498, 0.05612, 0.05730, 0.05851,
    0.05976, 0.06103, 0.06231, 0.06362, 0.06493, 0.06625, 0.06758, 0.06889,
    0.07020, 0.07149, 0.07277, 0.07402, 0.07524, 0.07644, 0.07760, 0.07872,
    0.07979, 0.08082, 0.08180, 0.08272, 0.08359, 0.08440, 0.08515, 0.08585,
    0.08649, 0.08708, 0.08761, 0.08808, 0.08850, 0.08886, 0.08916, 0.08941,
    0.08959, 0.08972, 0.08978, 0.08978, 0.08972, 0.08959, 0.08940, 0.08914,
    0.08882, 0.08843, 0.08797, 0.08745, 0.08687, 0.08622, 0.08551, 0.08474,
    0.08390, 0.08300, 0.08203, 0.08101, 0.07991, 0.07876, 0.07752, 0.07622,
    0.07485, 0.07341, 0.07190, 0.07031, 0.06865, 0.06692, 0.06511, 0.06324,
    0.06130, 0.05929, 0.05723, 0.05509, 0.05290, 0.05064, 0.04834, 0.04597,
    0.04355, 0.04109, 0.03857, 0.03602, 0.03342, 0.03077, 0.02810, 0.02539,
    0.02264, 0.01988, 0.01708, 0.01427, 0.01144, 0.00858, 0.00573, 0.00287,
    0.00000
};

const fastf_t EPS5[] = {
    0.00000, 0.00169, 0.00338, 0.00506, 0.00672, 0.00835, 0.00995, 0.01152,
    0.01304, 0.01450, 0.01591, 0.01719, 0.01822, 0.01896, 0.01952, 0.02004,
    0.02060, 0.02116, 0.02170, 0.02220, 0.02259, 0.02287, 0.02300, 0.02300,
    0.02289, 0.02270, 0.02245, 0.02215, 0.02184, 0.02154, 0.02126, 0.02101,
    0.02079, 0.02061, 0.02045, 0.02030, 0.02017, 0.02005, 0.01994, 0.01985,
    0.01979, 0.01972, 0.01969, 0.01966, 0.01966, 0.01968, 0.01971, 0.01975,
    0.01982, 0.01989, 0.01997, 0.02006, 0.02016, 0.02028, 0.02040, 0.02054,
    0.02067, 0.02083, 0.02099, 0.02116, 0.02135, 0.02155, 0.02175, 0.02198,
    0.02221, 0.02245, 0.02271, 0.02298, 0.02325, 0.02355, 0.02386, 0.02417,
    0.02450, 0.02484, 0.02519, 0.02555, 0.02592, 0.02630, 0.02670, 0.02710,
    0.02752, 0.02795, 0.02839, 0.02884, 0.02929, 0.02976, 0.03025, 0.03075,
    0.03126, 0.03178, 0.03232, 0.03287, 0.03343, 0.03401, 0.03461, 0.03522,
    0.03585, 0.03648, 0.03714, 0.03782, 0.03849, 0.03920, 0.03991, 0.04063,
    0.04136, 0.04213, 0.04290, 0.04369, 0.04451, 0.04535, 0.04622, 0.04712,
    0.04803, 0.04898, 0.04994, 0.05092, 0.05192, 0.05294, 0.05397, 0.05503,
    0.05610, 0.05721, 0.05833, 0.05949, 0.06067, 0.06187, 0.06309, 0.06434,
    0.06559, 0.06687, 0.06814, 0.06944, 0.07073, 0.07201, 0.07327, 0.07450,
    0.07571, 0.07689, 0.07801, 0.07908, 0.08009, 0.08104, 0.08194, 0.08277,
    0.08353, 0.08422, 0.08483, 0.08538, 0.08584, 0.08625, 0.08659, 0.08685,
    0.08706, 0.08720, 0.08729, 0.08732, 0.08728, 0.08716, 0.08691, 0.08654,
    0.08602, 0.08530, 0.08440, 0.08327, 0.08189, 0.08025, 0.07838, 0.07630,
    0.07403, 0.07162, 0.06908, 0.06647, 0.06378, 0.06107, 0.05836, 0.05568,
    0.05306, 0.05048, 0.04794, 0.04546, 0.04301, 0.04061, 0.03825, 0.03592,
    0.03363, 0.03137, 0.02914, 0.02694, 0.02477, 0.02262, 0.02049, 0.01838,
    0.01630, 0.01422, 0.01216, 0.01011, 0.00808, 0.00605, 0.00403, 0.00201,
    0.00000
};

const fastf_t EPS6[] = {
    0.00000, 0.00149, 0.00297, 0.00444, 0.00589, 0.00734, 0.00875, 0.01013,
    0.01148, 0.01279, 0.01405, 0.01512, 0.01582, 0.01616, 0.01631, 0.01642,
    0.01659, 0.01681, 0.01705, 0.01729, 0.01751, 0.01767, 0.01778, 0.01784,
    0.01787, 0.01787, 0.01788, 0.01788, 0.01793, 0.01800, 0.01813, 0.01831,
    0.01854, 0.01881, 0.01912, 0.01945, 0.01981, 0.02018, 0.02056, 0.02095,
    0.02134, 0.02175, 0.02216, 0.02257, 0.02299, 0.02342, 0.02385, 0.02429,
    0.02475, 0.02521, 0.02569, 0.02619, 0.02670, 0.02722, 0.02777, 0.02832,
    0.02889, 0.02947, 0.03005, 0.03065, 0.03125, 0.03186, 0.03247, 0.03308,
    0.03370, 0.03432, 0.03495, 0.03559, 0.03625, 0.03692, 0.03759, 0.03828,
    0.03898, 0.03969, 0.04041, 0.04114, 0.04189, 0.04265, 0.04341, 0.04419,
    0.04496, 0.04574, 0.04651, 0.04729, 0.04804, 0.04879, 0.04953, 0.05026,
    0.05100, 0.05171, 0.05243, 0.05314, 0.05386, 0.05456, 0.05525, 0.05594,
    0.05661, 0.05728, 0.05793, 0.05858, 0.05921, 0.05981, 0.06039, 0.06096,
    0.06149, 0.06200, 0.06248, 0.06295, 0.06339, 0.06381, 0.06422, 0.06461,
    0.06498, 0.06533, 0.06566, 0.06597, 0.06626, 0.06651, 0.06675, 0.06695,
    0.06714, 0.06729, 0.06740, 0.06749, 0.06754, 0.06755, 0.06754, 0.06751,
    0.06743, 0.06734, 0.06720, 0.06704, 0.06686, 0.06666, 0.06642, 0.06615,
    0.06586, 0.06554, 0.06520, 0.06481, 0.06441, 0.06397, 0.06351, 0.06303,
    0.06252, 0.06198, 0.06142, 0.06083, 0.06022, 0.05960, 0.05896, 0.05830,
    0.05763, 0.05695, 0.05626, 0.05555, 0.05482, 0.05406, 0.05325, 0.05241,
    0.05152, 0.05057, 0.04956, 0.04848, 0.04734, 0.04615, 0.04490, 0.04362,
    0.04231, 0.04095, 0.03959, 0.03820, 0.03681, 0.03542, 0.03404, 0.03266,
    0.03129, 0.02993, 0.02857, 0.02723, 0.02589, 0.02455, 0.02322, 0.02190,
    0.02058, 0.01927, 0.01797, 0.01667, 0.01537, 0.01408, 0.01279, 0.01150,
    0.01021, 0.00893, 0.00765, 0.00637, 0.00509, 0.00382, 0.00254, 0.00127,
    0.00000
};

const fastf_t EPS7[] = {
    0.00000, 0.00159, 0.00318, 0.00476, 0.00632, 0.00786, 0.00937, 0.01086,
    0.01231, 0.01372, 0.01507, 0.01623, 0.01702, 0.01743, 0.01761, 0.01776,
    0.01797, 0.01823, 0.01851, 0.01877, 0.01899, 0.01914, 0.01920, 0.01918,
    0.01911, 0.01898, 0.01885, 0.01871, 0.01858, 0.01850, 0.01846, 0.01848,
    0.01855, 0.01867, 0.01882, 0.01900, 0.01919, 0.01940, 0.01962, 0.01986,
    0.02010, 0.02036, 0.02062, 0.02088, 0.02115, 0.02144, 0.02174, 0.02205,
    0.02237, 0.02271, 0.02306, 0.02343, 0.02382, 0.02423, 0.02465, 0.02509,
    0.02555, 0.02602, 0.02650, 0.02699, 0.02749, 0.02800, 0.02850, 0.02902,
    0.02956, 0.03009, 0.03065, 0.03121, 0.03177, 0.03236, 0.03295, 0.03357,
    0.03420, 0.03484, 0.03550, 0.03618, 0.03688, 0.03760, 0.03833, 0.03908,
    0.03984, 0.04061, 0.04139, 0.04217, 0.04295, 0.04374, 0.04453, 0.04532,
    0.04612, 0.04691, 0.04772, 0.04852, 0.04932, 0.05013, 0.05094, 0.05172,
    0.05252, 0.05329, 0.05406, 0.05482, 0.05554, 0.05626, 0.05695, 0.05761,
    0.05824, 0.05884, 0.05942, 0.05997, 0.06049, 0.06099, 0.06147, 0.06193,
    0.06237, 0.06280, 0.06320, 0.06357, 0.06393, 0.06426, 0.06456, 0.06484,
    0.06508, 0.06530, 0.06548, 0.06564, 0.06576, 0.06585, 0.06591, 0.06595,
    0.06594, 0.06592, 0.06586, 0.06578, 0.06567, 0.06553, 0.06537, 0.06517,
    0.06496, 0.06470, 0.06442, 0.06410, 0.06376, 0.06339, 0.06299, 0.06257,
    0.06210, 0.06161, 0.06110, 0.06056, 0.06000, 0.05942, 0.05881, 0.05819,
    0.05756, 0.05691, 0.05625, 0.05557, 0.05487, 0.05413, 0.05336, 0.05254,
    0.05168, 0.05075, 0.04977, 0.04871, 0.04760, 0.04643, 0.04521, 0.04395,
    0.04265, 0.04132, 0.03997, 0.03860, 0.03722, 0.03584, 0.03446, 0.03309,
    0.03172, 0.03036, 0.02900, 0.02765, 0.02630, 0.02496, 0.02362, 0.02229,
    0.02095, 0.01963, 0.01831, 0.01699, 0.01567, 0.01436, 0.01304, 0.01173,
    0.01042, 0.00912, 0.00781, 0.00651, 0.00521, 0.00390, 0.00260, 0.00130,
    0.00000
};

const fastf_t EPS8[] = {
    0.00000, 0.00148, 0.00296, 0.00443, 0.00588, 0.00733, 0.00874, 0.01014,
    0.01151, 0.01284, 0.01413, 0.01523, 0.01598, 0.01638, 0.01659, 0.01676,
    0.01700, 0.01728, 0.01756, 0.01783, 0.01804, 0.01816, 0.01819, 0.01811,
    0.01796, 0.01775, 0.01752, 0.01724, 0.01695, 0.01668, 0.01642, 0.01619,
    0.01599, 0.01581, 0.01567, 0.01556, 0.01548, 0.01543, 0.01541, 0.01541,
    0.01545, 0.01551, 0.01559, 0.01570, 0.01583, 0.01598, 0.01616, 0.01635,
    0.01657, 0.01682, 0.01708, 0.01737, 0.01769, 0.01802, 0.01837, 0.01874,
    0.01913, 0.01952, 0.01992, 0.02033, 0.02075, 0.02115, 0.02157, 0.02199,
    0.02241, 0.02285, 0.02329, 0.02376, 0.02423, 0.02472, 0.02522, 0.02575,
    0.02629, 0.02685, 0.02742, 0.02802, 0.02863, 0.02926, 0.02990, 0.03055,
    0.03122, 0.03188, 0.03254, 0.03322, 0.03388, 0.03455, 0.03523, 0.03591,
    0.03661, 0.03732, 0.03806, 0.03881, 0.03958, 0.04036, 0.04115, 0.04197,
    0.04278, 0.04361, 0.04445, 0.04527, 0.04610, 0.04692, 0.04772, 0.04851,
    0.04928, 0.05004, 0.05077, 0.05149, 0.05219, 0.05288, 0.05356, 0.05422,
    0.05487, 0.05550, 0.05612, 0.05670, 0.05728, 0.05782, 0.05834, 0.05884,
    0.05931, 0.05975, 0.06015, 0.06053, 0.06086, 0.06117, 0.06144, 0.06169,
    0.06190, 0.06210, 0.06227, 0.06241, 0.06254, 0.06263, 0.06271, 0.06275,
    0.06277, 0.06275, 0.06271, 0.06264, 0.06254, 0.06240, 0.06222, 0.06201,
    0.06176, 0.06147, 0.06115, 0.06077, 0.06037, 0.05994, 0.05948, 0.05899,
    0.05849, 0.05797, 0.05741, 0.05685, 0.05625, 0.05562, 0.05494, 0.05421,
    0.05343, 0.05260, 0.05169, 0.05072, 0.04968, 0.04857, 0.04740, 0.04619,
    0.04494, 0.04365, 0.04234, 0.04098, 0.03962, 0.03824, 0.03685, 0.03546,
    0.03407, 0.03268, 0.03128, 0.02988, 0.02847, 0.02706, 0.02565, 0.02424,
    0.02282, 0.02141, 0.01999, 0.01857, 0.01715, 0.01573, 0.01430, 0.01287,
    0.01144, 0.01002, 0.00859, 0.00716, 0.00573, 0.00430, 0.00286, 0.00143,
    0.00000
};

const fastf_t PSI1[] = {
    0.15066, 0.15058, 0.15035, 0.14999, 0.14950, 0.14891, 0.14823, 0.14748,
    0.14668, 0.14583, 0.14497, 0.14410, 0.14323, 0.14238, 0.14155, 0.14074,
    0.13998, 0.13927, 0.13862, 0.13804, 0.13753, 0.13711, 0.13676, 0.13648,
    0.13627, 0.13610, 0.13598, 0.13590, 0.13584, 0.13579, 0.13576, 0.13573,
    0.13570, 0.13567, 0.13564, 0.13561, 0.13558, 0.13555, 0.13552, 0.13550,
    0.13547, 0.13544, 0.13542, 0.13539, 0.13536, 0.13533, 0.13529, 0.13525,
    0.13521, 0.13516, 0.13511, 0.13505, 0.13499, 0.13491, 0.13483, 0.13475,
    0.13465, 0.13454, 0.13442, 0.13428, 0.13414, 0.13398, 0.13381, 0.13363,
    0.13343, 0.13321, 0.13299, 0.13275, 0.13249, 0.13222, 0.13194, 0.13164,
    0.13133, 0.13100, 0.13065, 0.13028, 0.12988, 0.12947, 0.12903, 0.12857,
    0.12808, 0.12756, 0.12702, 0.12644, 0.12584, 0.12521, 0.12455, 0.12385,
    0.12313, 0.12238, 0.12160, 0.12079, 0.11994, 0.11907, 0.11817, 0.11724,
    0.11628, 0.11529, 0.11428, 0.11324, 0.11218, 0.11109, 0.10998, 0.10884,
    0.10768, 0.10650, 0.10530, 0.10407, 0.10283, 0.10157, 0.10029, 0.09899,
    0.09767, 0.09634, 0.09499, 0.09363, 0.09224, 0.09085, 0.08944, 0.08801,
    0.08657, 0.08512, 0.08365, 0.08217, 0.08068, 0.07917, 0.07766, 0.07614,
    0.07461, 0.07307, 0.07153, 0.06998, 0.06842, 0.06687, 0.06530, 0.06374,
    0.06217, 0.06060, 0.05904, 0.05747, 0.05591, 0.05435, 0.05280, 0.05125,
    0.04970, 0.04817, 0.04664, 0.04512, 0.04362, 0.04213, 0.04065, 0.03919,
    0.03774, 0.03631, 0.03490, 0.03350, 0.03213, 0.03077, 0.02943, 0.02811,
    0.02682, 0.02555, 0.02430, 0.02308, 0.02188, 0.02071, 0.01956, 0.01844,
    0.01735, 0.01630, 0.01527, 0.01428, 0.01331, 0.01239, 0.01149, 0.01062,
    0.00979, 0.00899, 0.00823, 0.00750, 0.00680, 0.00614, 0.00551, 0.00491,
    0.00435, 0.00382, 0.00332, 0.00286, 0.00244, 0.00205, 0.00169, 0.00137,
    0.00108, 0.00083, 0.00061, 0.00042, 0.00027, 0.00015, 0.00007, 0.00002,
    0.00000
};

const fastf_t PSI2[] = {
    0.25269, 0.25265, 0.25251, 0.25227, 0.25193, 0.25147, 0.25090, 0.25020,
    0.24937, 0.24841, 0.24730, 0.24605, 0.24467, 0.24321, 0.24170, 0.24016,
    0.23864, 0.23715, 0.23573, 0.23442, 0.23325, 0.23224, 0.23138, 0.23066,
    0.23006, 0.22956, 0.22916, 0.22884, 0.22858, 0.22836, 0.22818, 0.22802,
    0.22788, 0.22775, 0.22764, 0.22755, 0.22747, 0.22740, 0.22736, 0.22732,
    0.22730, 0.22729, 0.22730, 0.22731, 0.22733, 0.22736, 0.22739, 0.22742,
    0.22745, 0.22748, 0.22751, 0.22753, 0.22755, 0.22756, 0.22756, 0.22755,
    0.22753, 0.22751, 0.22747, 0.22742, 0.22736, 0.22729, 0.22720, 0.22709,
    0.22697, 0.22683, 0.22668, 0.22650, 0.22630, 0.22608, 0.22584, 0.22557,
    0.22528, 0.22497, 0.22462, 0.22426, 0.22386, 0.22345, 0.22300, 0.22253,
    0.22203, 0.22150, 0.22094, 0.22034, 0.21969, 0.21899, 0.21823, 0.21741,
    0.21652, 0.21554, 0.21449, 0.21334, 0.21211, 0.21081, 0.20941, 0.20795,
    0.20642, 0.20482, 0.20316, 0.20143, 0.19966, 0.19784, 0.19597, 0.19406,
    0.19210, 0.19009, 0.18805, 0.18597, 0.18385, 0.18169, 0.17950, 0.17727,
    0.17502, 0.17274, 0.17042, 0.16807, 0.16570, 0.16330, 0.16087, 0.15843,
    0.15596, 0.15347, 0.15095, 0.14842, 0.14588, 0.14331, 0.14073, 0.13812,
    0.13551, 0.13288, 0.13024, 0.12759, 0.12492, 0.12225, 0.11957, 0.11688,
    0.11418, 0.11149, 0.10878, 0.10608, 0.10338, 0.10068, 0.09798, 0.09529,
    0.09260, 0.08992, 0.08725, 0.08459, 0.08195, 0.07932, 0.07671, 0.07412,
    0.07155, 0.06900, 0.06648, 0.06398, 0.06151, 0.05907, 0.05666, 0.05428,
    0.05193, 0.04962, 0.04733, 0.04509, 0.04288, 0.04072, 0.03859, 0.03650,
    0.03446, 0.03247, 0.03051, 0.02861, 0.02674, 0.02493, 0.02317, 0.02147,
    0.01982, 0.01824, 0.01672, 0.01525, 0.01385, 0.01252, 0.01125, 0.01006,
    0.00892, 0.00786, 0.00686, 0.00593, 0.00506, 0.00426, 0.00353, 0.00287,
    0.00227, 0.00174, 0.00128, 0.00089, 0.00057, 0.00032, 0.00014, 0.00004,
    0.00000
};

const fastf_t PSI3[] = {
    0.17464, 0.17456, 0.17434, 0.17397, 0.17348, 0.17285, 0.17211, 0.17125,
    0.17030, 0.16923, 0.16808, 0.16682, 0.16542, 0.16391, 0.16236, 0.16086,
    0.15949, 0.15823, 0.15711, 0.15610, 0.15523, 0.15451, 0.15390, 0.15343,
    0.15305, 0.15278, 0.15258, 0.15246, 0.15239, 0.15235, 0.15235, 0.15237,
    0.15241, 0.15249, 0.15258, 0.15269, 0.15282, 0.15297, 0.15314, 0.15331,
    0.15350, 0.15370, 0.15389, 0.15408, 0.15427, 0.15445, 0.15464, 0.15482,
    0.15501, 0.15518, 0.15536, 0.15553, 0.15569, 0.15585, 0.15600, 0.15615,
    0.15629, 0.15643, 0.15655, 0.15667, 0.15678, 0.15688, 0.15697, 0.15705,
    0.15711, 0.15718, 0.15723, 0.15726, 0.15729, 0.15731, 0.15731, 0.15730,
    0.15728, 0.15724, 0.15719, 0.15713, 0.15704, 0.15694, 0.15682, 0.15669,
    0.15653, 0.15637, 0.15618, 0.15598, 0.15576, 0.15552, 0.15526, 0.15498,
    0.15466, 0.15431, 0.15393, 0.15351, 0.15305, 0.15255, 0.15200, 0.15141,
    0.15078, 0.15011, 0.14938, 0.14862, 0.14779, 0.14693, 0.14602, 0.14505,
    0.14402, 0.14295, 0.14181, 0.14063, 0.13940, 0.13812, 0.13680, 0.13544,
    0.13403, 0.13259, 0.13110, 0.12958, 0.12804, 0.12646, 0.12484, 0.12320,
    0.12154, 0.11984, 0.11813, 0.11638, 0.11461, 0.11281, 0.11100, 0.10915,
    0.10730, 0.10541, 0.10353, 0.10161, 0.09970, 0.09778, 0.09583, 0.09389,
    0.09194, 0.08998, 0.08800, 0.08601, 0.08401, 0.08201, 0.08000, 0.07798,
    0.07596, 0.07393, 0.07191, 0.06989, 0.06788, 0.06586, 0.06385, 0.06185,
    0.05986, 0.05787, 0.05589, 0.05392, 0.05196, 0.05001, 0.04807, 0.04615,
    0.04422, 0.04232, 0.04042, 0.03854, 0.03667, 0.03481, 0.03297, 0.03115,
    0.02937, 0.02762, 0.02590, 0.02423, 0.02260, 0.02101, 0.01949, 0.01802,
    0.01661, 0.01526, 0.01396, 0.01272, 0.01154, 0.01042, 0.00935, 0.00834,
    0.00739, 0.00650, 0.00566, 0.00489, 0.00416, 0.00350, 0.00289, 0.00234,
    0.00185, 0.00141, 0.00104, 0.00072, 0.00046, 0.00026, 0.00012, 0.00003,
    0.00000
};

const fastf_t PSI4[] = {
    0.16457, 0.16455, 0.16449, 0.16437, 0.16416, 0.16386, 0.16345, 0.16292,
    0.16223, 0.16139, 0.16037, 0.15916, 0.15779, 0.15631, 0.15475, 0.15316,
    0.15157, 0.15002, 0.14856, 0.14722, 0.14604, 0.14506, 0.14427, 0.14364,
    0.14316, 0.14281, 0.14257, 0.14242, 0.14235, 0.14233, 0.14236, 0.14241,
    0.14248, 0.14257, 0.14267, 0.14280, 0.14294, 0.14310, 0.14327, 0.14346,
    0.14366, 0.14387, 0.14410, 0.14433, 0.14457, 0.14481, 0.14506, 0.14530,
    0.14554, 0.14578, 0.14601, 0.14623, 0.14645, 0.14665, 0.14685, 0.14704,
    0.14722, 0.14740, 0.14757, 0.14774, 0.14790, 0.14806, 0.14821, 0.14835,
    0.14849, 0.14862, 0.14875, 0.14886, 0.14897, 0.14908, 0.14917, 0.14925,
    0.14933, 0.14940, 0.14945, 0.14950, 0.14954, 0.14957, 0.14959, 0.14961,
    0.14961, 0.14960, 0.14959, 0.14956, 0.14953, 0.14948, 0.14943, 0.14936,
    0.14928, 0.14918, 0.14908, 0.14896, 0.14883, 0.14869, 0.14853, 0.14835,
    0.14816, 0.14796, 0.14774, 0.14750, 0.14725, 0.14698, 0.14669, 0.14638,
    0.14606, 0.14571, 0.14533, 0.14494, 0.14452, 0.14407, 0.14360, 0.14310,
    0.14256, 0.14198, 0.14135, 0.14067, 0.13992, 0.13910, 0.13820, 0.13722,
    0.13615, 0.13498, 0.13371, 0.13237, 0.13094, 0.12943, 0.12787, 0.12624,
    0.12457, 0.12284, 0.12108, 0.11929, 0.11746, 0.11561, 0.11374, 0.11183,
    0.10988, 0.10790, 0.10590, 0.10386, 0.10180, 0.09970, 0.09758, 0.09542,
    0.09325, 0.09106, 0.08885, 0.08662, 0.08439, 0.08214, 0.07989, 0.07763,
    0.07537, 0.07311, 0.07085, 0.06859, 0.06633, 0.06407, 0.06182, 0.05957,
    0.05733, 0.05510, 0.05287, 0.05066, 0.04845, 0.04627, 0.04411, 0.04196,
    0.03983, 0.03773, 0.03566, 0.03362, 0.03160, 0.02963, 0.02769, 0.02579,
    0.02394, 0.02214, 0.02038, 0.01869, 0.01705, 0.01547, 0.01396, 0.01249,
    0.01111, 0.00980, 0.00856, 0.00740, 0.00631, 0.00531, 0.00439, 0.00356,
    0.00281, 0.00216, 0.00159, 0.00110, 0.00071, 0.00040, 0.00018, 0.00004,
    0.00000
};

const fastf_t PSI5[] = {
    0.18028, 0.18026, 0.18017, 0.17999, 0.17970, 0.17927, 0.17866, 0.17786,
    0.17684, 0.17558, 0.17403, 0.17233, 0.17077, 0.16944, 0.16824, 0.16708,
    0.16586, 0.16461, 0.16334, 0.16208, 0.16085, 0.15968, 0.15859, 0.15759,
    0.15667, 0.15583, 0.15506, 0.15438, 0.15377, 0.15323, 0.15278, 0.15240,
    0.15207, 0.15179, 0.15156, 0.15135, 0.15115, 0.15097, 0.15079, 0.15063,
    0.15048, 0.15035, 0.15023, 0.15012, 0.15004, 0.14997, 0.14992, 0.14988,
    0.14985, 0.14983, 0.14982, 0.14982, 0.14982, 0.14983, 0.14985, 0.14986,
    0.14989, 0.14992, 0.14995, 0.14999, 0.15002, 0.15006, 0.15010, 0.15014,
    0.15019, 0.15023, 0.15027, 0.15032, 0.15036, 0.15041, 0.15046, 0.15050,
    0.15056, 0.15060, 0.15066, 0.15070, 0.15076, 0.15081, 0.15085, 0.15090,
    0.15095, 0.15098, 0.15101, 0.15103, 0.15106, 0.15108, 0.15108, 0.15109,
    0.15110, 0.15110, 0.15110, 0.15110, 0.15109, 0.15108, 0.15106, 0.15103,
    0.15100, 0.15096, 0.15090, 0.15084, 0.15076, 0.15068, 0.15059, 0.15050,
    0.15038, 0.15026, 0.15014, 0.14999, 0.14983, 0.14965, 0.14945, 0.14924,
    0.14900, 0.14875, 0.14847, 0.14817, 0.14784, 0.14750, 0.14712, 0.14671,
    0.14628, 0.14582, 0.14532, 0.14479, 0.14422, 0.14362, 0.14296, 0.14226,
    0.14149, 0.14065, 0.13974, 0.13875, 0.13768, 0.13652, 0.13526, 0.13391,
    0.13246, 0.13090, 0.12925, 0.12748, 0.12560, 0.12366, 0.12163, 0.11954,
    0.11738, 0.11520, 0.11298, 0.11074, 0.10848, 0.10618, 0.10386, 0.10150,
    0.09911, 0.09667, 0.09420, 0.09168, 0.08911, 0.08650, 0.08384, 0.08115,
    0.07840, 0.07562, 0.07280, 0.06994, 0.06704, 0.06412, 0.06117, 0.05820,
    0.05524, 0.05228, 0.04935, 0.04644, 0.04358, 0.04078, 0.03803, 0.03536,
    0.03277, 0.03026, 0.02785, 0.02551, 0.02327, 0.02112, 0.01906, 0.01710,
    0.01522, 0.01345, 0.01178, 0.01021, 0.00875, 0.00740, 0.00615, 0.00501,
    0.00397, 0.00306, 0.00226, 0.00158, 0.00101, 0.00057, 0.00025, 0.00006,
    0.00000
};

const fastf_t PSI6[] = {
    0.15517, 0.15505, 0.15472, 0.15418, 0.15346, 0.15257, 0.15154, 0.15038,
    0.14911, 0.14776, 0.14634, 0.14493, 0.14363, 0.14249, 0.14146, 0.14058,
    0.13979, 0.13911, 0.13851, 0.13799, 0.13753, 0.13713, 0.13678, 0.13647,
    0.13620, 0.13596, 0.13576, 0.13559, 0.13543, 0.13531, 0.13521, 0.13512,
    0.13505, 0.13499, 0.13493, 0.13488, 0.13484, 0.13480, 0.13476, 0.13471,
    0.13468, 0.13463, 0.13460, 0.13456, 0.13452, 0.13447, 0.13442, 0.13438,
    0.13432, 0.13427, 0.13420, 0.13414, 0.13406, 0.13399, 0.13390, 0.13380,
    0.13371, 0.13360, 0.13348, 0.13336, 0.13322, 0.13307, 0.13292, 0.13276,
    0.13258, 0.13239, 0.13220, 0.13198, 0.13177, 0.13154, 0.13131, 0.13105,
    0.13079, 0.13051, 0.13020, 0.12989, 0.12955, 0.12917, 0.12878, 0.12836,
    0.12792, 0.12747, 0.12698, 0.12648, 0.12595, 0.12540, 0.12482, 0.12423,
    0.12361, 0.12295, 0.12228, 0.12158, 0.12085, 0.12009, 0.11932, 0.11851,
    0.11768, 0.11683, 0.11596, 0.11507, 0.11416, 0.11322, 0.11227, 0.11129,
    0.11030, 0.10928, 0.10825, 0.10720, 0.10614, 0.10505, 0.10395, 0.10284,
    0.10171, 0.10057, 0.09941, 0.09825, 0.09706, 0.09586, 0.09466, 0.09344,
    0.09221, 0.09096, 0.08972, 0.08845, 0.08718, 0.08590, 0.08462, 0.08332,
    0.08202, 0.08071, 0.07941, 0.07810, 0.07679, 0.07548, 0.07417, 0.07288,
    0.07158, 0.07029, 0.06902, 0.06775, 0.06650, 0.06525, 0.06402, 0.06278,
    0.06156, 0.06035, 0.05914, 0.05794, 0.05675, 0.05558, 0.05441, 0.05325,
    0.05210, 0.05096, 0.04983, 0.04872, 0.04761, 0.04650, 0.04539, 0.04428,
    0.04315, 0.04203, 0.04089, 0.03973, 0.03857, 0.03741, 0.03625, 0.03511,
    0.03399, 0.03291, 0.03186, 0.03086, 0.02991, 0.02902, 0.02820, 0.02745,
    0.02677, 0.02615, 0.02559, 0.02509, 0.02464, 0.02425, 0.02390, 0.02360,
    0.02333, 0.02311, 0.02293, 0.02277, 0.02265, 0.02256, 0.02248, 0.02244,
    0.02240, 0.02238, 0.02238, 0.02238, 0.02239, 0.02241, 0.02242, 0.02242,
    0.02243
};

const fastf_t PSI7[] = {
    0.15365, 0.15354, 0.15318, 0.15263, 0.15188, 0.15098, 0.14993, 0.14876,
    0.14748, 0.14614, 0.14473, 0.14330, 0.14190, 0.14057, 0.13933, 0.13819,
    0.13717, 0.13626, 0.13544, 0.13472, 0.13407, 0.13349, 0.13296, 0.13249,
    0.13208, 0.13170, 0.13137, 0.13107, 0.13081, 0.13059, 0.13038, 0.13019,
    0.13004, 0.12990, 0.12978, 0.12969, 0.12962, 0.12956, 0.12953, 0.12951,
    0.12949, 0.12949, 0.12949, 0.12948, 0.12947, 0.12947, 0.12946, 0.12945,
    0.12944, 0.12943, 0.12942, 0.12941, 0.12940, 0.12939, 0.12938, 0.12937,
    0.12936, 0.12934, 0.12932, 0.12929, 0.12925, 0.12921, 0.12915, 0.12909,
    0.12902, 0.12894, 0.12885, 0.12876, 0.12865, 0.12853, 0.12841, 0.12827,
    0.12812, 0.12797, 0.12781, 0.12762, 0.12743, 0.12723, 0.12701, 0.12677,
    0.12652, 0.12625, 0.12596, 0.12567, 0.12535, 0.12500, 0.12464, 0.12424,
    0.12380, 0.12333, 0.12281, 0.12224, 0.12164, 0.12099, 0.12030, 0.11958,
    0.11883, 0.11805, 0.11724, 0.11641, 0.11554, 0.11466, 0.11375, 0.11282,
    0.11186, 0.11088, 0.10988, 0.10885, 0.10781, 0.10675, 0.10568, 0.10459,
    0.10349, 0.10237, 0.10124, 0.10009, 0.09893, 0.09776, 0.09657, 0.09537,
    0.09416, 0.09294, 0.09170, 0.09045, 0.08919, 0.08793, 0.08664, 0.08537,
    0.08407, 0.08277, 0.08147, 0.08017, 0.07886, 0.07755, 0.07625, 0.07495,
    0.07365, 0.07235, 0.07105, 0.06976, 0.06848, 0.06720, 0.06592, 0.06466,
    0.06340, 0.06215, 0.06091, 0.05967, 0.05846, 0.05724, 0.05604, 0.05485,
    0.05368, 0.05250, 0.05135, 0.05020, 0.04906, 0.04792, 0.04678, 0.04564,
    0.04450, 0.04333, 0.04216, 0.04098, 0.03978, 0.03859, 0.03740, 0.03623,
    0.03508, 0.03396, 0.03288, 0.03184, 0.03085, 0.02992, 0.02906, 0.02827,
    0.02754, 0.02688, 0.02628, 0.02573, 0.02524, 0.02480, 0.02440, 0.02406,
    0.02375, 0.02348, 0.02326, 0.02306, 0.02290, 0.02277, 0.02266, 0.02259,
    0.02252, 0.02247, 0.02245, 0.02242, 0.02242, 0.02242, 0.02242, 0.02242,
    0.02243
};

const fastf_t PSI8[] = {
    0.14761, 0.14746, 0.14701, 0.14632, 0.14542, 0.14435, 0.14314, 0.14185,
    0.14052, 0.13916, 0.13785, 0.13662, 0.13554, 0.13460, 0.13376, 0.13296,
    0.13216, 0.13136, 0.13056, 0.12974, 0.12892, 0.12808, 0.12722, 0.12638,
    0.12555, 0.12476, 0.12406, 0.12342, 0.12288, 0.12247, 0.12219, 0.12204,
    0.12199, 0.12201, 0.12210, 0.12222, 0.12235, 0.12246, 0.12258, 0.12268,
    0.12279, 0.12289, 0.12300, 0.12310, 0.12321, 0.12331, 0.12343, 0.12354,
    0.12365, 0.12376, 0.12387, 0.12397, 0.12407, 0.12417, 0.12426, 0.12435,
    0.12443, 0.12451, 0.12459, 0.12466, 0.12472, 0.12478, 0.12484, 0.12488,
    0.12492, 0.12496, 0.12499, 0.12501, 0.12502, 0.12503, 0.12502, 0.12501,
    0.12499, 0.12495, 0.12491, 0.12487, 0.12481, 0.12474, 0.12466, 0.12457,
    0.12448, 0.12436, 0.12424, 0.12410, 0.12394, 0.12377, 0.12358, 0.12338,
    0.12316, 0.12292, 0.12266, 0.12237, 0.12206, 0.12174, 0.12137, 0.12099,
    0.12058, 0.12013, 0.11965, 0.11914, 0.11859, 0.11801, 0.11740, 0.11675,
    0.11607, 0.11534, 0.11459, 0.11379, 0.11297, 0.11211, 0.11122, 0.11031,
    0.10937, 0.10840, 0.10741, 0.10639, 0.10536, 0.10431, 0.10324, 0.10217,
    0.10107, 0.09995, 0.09881, 0.09766, 0.09650, 0.09532, 0.09412, 0.09292,
    0.09170, 0.09046, 0.08921, 0.08796, 0.08668, 0.08541, 0.08412, 0.08282,
    0.08151, 0.08020, 0.07889, 0.07755, 0.07622, 0.07489, 0.07354, 0.07220,
    0.07084, 0.06949, 0.06813, 0.06678, 0.06543, 0.06408, 0.06275, 0.06141,
    0.06009, 0.05879, 0.05749, 0.05621, 0.05494, 0.05367, 0.05240, 0.05113,
    0.04985, 0.04857, 0.04727, 0.04597, 0.04465, 0.04333, 0.04202, 0.04072,
    0.03943, 0.03818, 0.03695, 0.03577, 0.03463, 0.03355, 0.03252, 0.03156,
    0.03067, 0.02982, 0.02904, 0.02831, 0.02763, 0.02702, 0.02644, 0.02592,
    0.02544, 0.02500, 0.02461, 0.02426, 0.02394, 0.02365, 0.02341, 0.02320,
    0.02302, 0.02286, 0.02274, 0.02263, 0.02255, 0.02250, 0.02245, 0.02243,
    0.02242
};

/*
 * Local Variables:
 * tab-width: 8
 * mode: C
 * indent-tabs-mode: t
 * c-file-style: "stroustrup"
 * End:
 * ex: shiftwidth=4 tabstop=8
 */
