/*
 * Decompiled with CFR 0.152.
 */
package org.destinationsol.game.console;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Queues;
import com.google.common.collect.Sets;
import com.google.common.collect.UnmodifiableIterator;
import com.google.common.primitives.Primitives;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayDeque;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.destinationsol.game.SolGame;
import org.destinationsol.game.console.CommandParameter;
import org.destinationsol.game.console.ConsoleCommand;
import org.destinationsol.game.console.MarkerParameters;
import org.destinationsol.game.console.Parameter;
import org.destinationsol.game.console.exceptions.CommandExecutionException;
import org.destinationsol.game.console.exceptions.CommandInitializationException;
import org.destinationsol.game.console.exceptions.CommandParameterParseException;
import org.destinationsol.game.console.exceptions.CommandSuggestionException;
import org.destinationsol.game.context.Context;
import org.destinationsol.util.SpecificAccessibleObject;

public abstract class AbstractCommand
implements ConsoleCommand {
    private final String name;
    private final String description;
    private final String helpText;
    private final SpecificAccessibleObject<Method> executionMethod;
    private ImmutableList<CommandParameter> commandParameters;
    private ImmutableList<Parameter> executionMethodParameters;
    private int requiredParameterCount;
    private String usage;
    private SolGame game;

    public AbstractCommand(String name, String description, String helpText, SpecificAccessibleObject<Method> executionMethod, SolGame game, Context context) {
        Preconditions.checkNotNull(executionMethod);
        Preconditions.checkNotNull((Object)description);
        Preconditions.checkNotNull((Object)helpText);
        this.name = name;
        this.description = description;
        this.helpText = helpText;
        this.executionMethod = executionMethod;
        this.game = game;
        this.constructParametersNotNull(context);
        this.registerParameters();
        this.validateExecutionMethod();
        this.initUsage();
    }

    private static Set<String> convertToString(Set<Object> collection, CommandParameter parameter) {
        return collection.stream().map(parameter::convertToString).collect(Collectors.toCollection(HashSet::new));
    }

    protected abstract List<Parameter> constructParameters(Context var1);

    private void constructParametersNotNull(Context context) {
        List<Parameter> constructedParameters = this.constructParameters(context);
        if (constructedParameters == null || constructedParameters.size() <= 0) {
            this.commandParameters = ImmutableList.of();
            this.executionMethodParameters = ImmutableList.of();
            return;
        }
        ImmutableList.Builder commandParameterBuilder = ImmutableList.builder();
        for (int i = 0; i < constructedParameters.size(); ++i) {
            Parameter type = constructedParameters.get(i);
            if (type == null) {
                throw new CommandInitializationException("Invalid parameter definition #" + i + "; must not be null");
            }
            if (!(type instanceof CommandParameter)) continue;
            commandParameterBuilder.add((Object)((CommandParameter)type));
        }
        this.commandParameters = commandParameterBuilder.build();
        this.executionMethodParameters = ImmutableList.copyOf(constructedParameters);
    }

    private void registerParameters() throws CommandInitializationException {
        this.requiredParameterCount = 0;
        boolean optionalFound = false;
        for (int i = 0; i < this.commandParameters.size(); ++i) {
            CommandParameter parameter = (CommandParameter)this.commandParameters.get(i);
            if (parameter == null) {
                throw new CommandInitializationException("A command parameter must not be null! Index: " + i);
            }
            if (parameter.isArray() && i < this.commandParameters.size() - 1) {
                throw new CommandInitializationException("A varargs parameter must be at the end. Invalid: " + i + "; " + parameter.getName());
            }
            if (parameter.isRequired()) {
                if (!optionalFound) {
                    ++this.requiredParameterCount;
                    continue;
                }
                throw new CommandInitializationException("A command definition must not contain a required parameter (" + i + "; " + parameter.getName() + ") after an optional parameter.");
            }
            if (optionalFound) continue;
            optionalFound = true;
        }
    }

    private void checkArgumentCompatibility(Method method) throws CommandInitializationException {
        Class<?>[] methodParameters = method.getParameterTypes();
        int executionMethodParametersSize = this.executionMethodParameters.size();
        int methodParameterCount = methodParameters.length;
        for (int i = 0; i < methodParameterCount || i < executionMethodParametersSize; ++i) {
            if (i >= methodParameterCount) {
                throw new CommandInitializationException("Missing " + (executionMethodParametersSize - methodParameterCount) + " parameters in method " + method.getName() + ", follow the parameter definitions from the 'constructParameters' method.");
            }
            if (i >= executionMethodParametersSize) {
                throw new CommandInitializationException("Too many (" + (methodParameterCount - executionMethodParametersSize) + ") parameters in method " + method.getName() + ", follow the parameter definitions from the 'constructParameters' method.");
            }
            Parameter expectedParameterType = (Parameter)this.executionMethodParameters.get(i);
            Optional<Class<?>> expectedType = expectedParameterType.getProvidedType();
            Class providedType = methodParameters[i];
            if (providedType.isPrimitive()) {
                providedType = Primitives.wrap(providedType);
            }
            if (!expectedType.isPresent() || expectedType.get().isAssignableFrom(providedType)) continue;
            throw new CommandInitializationException("Cannot assign command argument from " + providedType.getSimpleName() + " to " + expectedType.get().getSimpleName() + "; command method parameter index: " + i);
        }
    }

    private void validateExecutionMethod() {
        this.checkArgumentCompatibility(this.executionMethod.getAccessibleObject());
    }

    private void initUsage() {
        StringBuilder builder = new StringBuilder(this.name);
        for (CommandParameter param : this.commandParameters) {
            builder.append(' ').append(param.getUsage());
        }
        this.usage = builder.toString();
    }

    private Object[] processParametersMethod(List<String> rawParameters) throws CommandParameterParseException {
        Object[] processedParameters = new Object[this.executionMethodParameters.size()];
        ArrayDeque parameterStrings = Queues.newArrayDeque(rawParameters);
        for (int i = 0; i < this.executionMethodParameters.size(); ++i) {
            Parameter parameterType = (Parameter)this.executionMethodParameters.get(i);
            if (parameterType instanceof CommandParameter) {
                CommandParameter parameter = (CommandParameter)parameterType;
                if (parameterStrings.isEmpty()) {
                    if (parameter.isArray()) {
                        processedParameters[i] = parameter.getArrayValue(Collections.emptyList());
                        continue;
                    }
                    processedParameters[i] = null;
                    continue;
                }
                if (parameter.isArray()) {
                    processedParameters[i] = parameter.getArrayValue(Lists.newArrayList((Iterable)parameterStrings));
                    parameterStrings.clear();
                    continue;
                }
                processedParameters[i] = parameter.getValue((String)parameterStrings.poll());
                continue;
            }
            if (parameterType != MarkerParameters.GAME) continue;
            processedParameters[i] = this.game;
        }
        return processedParameters;
    }

    @Override
    public final String execute(List<String> rawParameters) throws CommandExecutionException {
        Object[] processedParameters;
        try {
            processedParameters = this.processParametersMethod(rawParameters);
        }
        catch (CommandParameterParseException e) {
            String warning = "Invalid parameter '" + e.getParameter() + "'";
            String message = e.getMessage();
            if (message != null) {
                warning = warning + ": " + message;
            }
            return warning;
        }
        try {
            Object result = this.executionMethod.getAccessibleObject().invoke(this.executionMethod.getTarget(), processedParameters);
            return result != null ? String.valueOf(result) : null;
        }
        catch (InvocationTargetException t) {
            if (t.getCause() != null) {
                throw new CommandExecutionException(t.getCause());
            }
            throw new CommandExecutionException(t);
        }
        catch (IllegalAccessException | RuntimeException t) {
            throw new CommandExecutionException(t);
        }
    }

    @Override
    public final Set<String> suggest(String currentValue, List<String> rawParameters) throws CommandSuggestionException {
        Object[] processedParameters;
        try {
            processedParameters = this.processParametersMethod(rawParameters);
        }
        catch (CommandParameterParseException e) {
            String warning = "Invalid parameter '" + e.getParameter() + "'";
            String message = e.getMessage();
            if (message != null) {
                warning = warning + ": " + message;
            }
            throw new CommandSuggestionException(warning);
        }
        CommandParameter suggestedParameter = null;
        UnmodifiableIterator paramIter = this.commandParameters.iterator();
        for (Object object : processedParameters) {
            if (this.game.equals(object)) continue;
            if (object == null) {
                suggestedParameter = (CommandParameter)paramIter.next();
                break;
            }
            paramIter.next();
        }
        if (suggestedParameter == null) {
            return Sets.newHashSet();
        }
        Set<Object> result = null;
        result = suggestedParameter.suggest(processedParameters);
        if (result == null) {
            return Sets.newHashSet();
        }
        Class<?> requiredClass = suggestedParameter.getType();
        for (Object t : result) {
            if (t == null && requiredClass.isPrimitive()) {
                throw new CommandSuggestionException("The 'suggest' method of command class " + this.getClass().getCanonicalName() + " returns a collection containing an invalid type. Required: " + requiredClass.getCanonicalName() + "; provided: null");
            }
            if (t == null || requiredClass.isAssignableFrom(t.getClass())) continue;
            throw new CommandSuggestionException("The 'suggest' method of command class " + this.getClass().getCanonicalName() + " returns a collection containing an invalid type. Required: " + requiredClass.getCanonicalName() + "; provided: " + t.getClass().getCanonicalName());
        }
        Set<String> stringSuggestions = AbstractCommand.convertToString(result, suggestedParameter);
        return Sets.filter(stringSuggestions, input -> input != null && (currentValue == null || input.startsWith(currentValue)));
    }

    @Override
    public ImmutableList<CommandParameter> getCommandParameters() {
        return this.commandParameters;
    }

    @Override
    public String getDescription() {
        return this.description;
    }

    @Override
    public String getHelpText() {
        return this.helpText;
    }

    @Override
    public String getUsage() {
        return this.usage;
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public int getRequiredParameterCount() {
        return this.requiredParameterCount;
    }

    @Override
    public boolean endsWithVarargs() {
        return this.commandParameters.size() > 0 && ((CommandParameter)this.commandParameters.get(this.commandParameters.size() - 1)).isArray();
    }

    @Override
    public Object getSource() {
        return this.executionMethod.getTarget();
    }

    @Override
    public int compareTo(ConsoleCommand o) {
        return ConsoleCommand.COMPARATOR.compare(this, o);
    }

    public SpecificAccessibleObject<Method> getExecutionMethod() {
        return this.executionMethod;
    }
}

