/*
 * Decompiled with CFR 0.152.
 */
package org.terasology.gestalt.assets;

import android.support.annotation.Nullable;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.MapMaker;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.IOException;
import java.lang.ref.PhantomReference;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.lang.reflect.Type;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Semaphore;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terasology.gestalt.assets.Asset;
import org.terasology.gestalt.assets.AssetData;
import org.terasology.gestalt.assets.AssetDataProducer;
import org.terasology.gestalt.assets.AssetFactory;
import org.terasology.gestalt.assets.DisposalHook;
import org.terasology.gestalt.assets.ResolutionStrategy;
import org.terasology.gestalt.assets.ResourceUrn;
import org.terasology.gestalt.module.sandbox.API;
import org.terasology.gestalt.naming.Name;
import org.terasology.gestalt.util.reflection.GenericsUtil;

@API
@ThreadSafe
public final class AssetType<T extends Asset<U>, U extends AssetData>
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(AssetType.class);
    private final Class<T> assetClass;
    private final Class<U> assetDataClass;
    private final AssetFactory<T, U> factory;
    private final List<AssetDataProducer<U>> producers = Lists.newCopyOnWriteArrayList();
    private final Map<ResourceUrn, T> loadedAssets = new MapMaker().concurrencyLevel(4).makeMap();
    private final ListMultimap<ResourceUrn, WeakReference<T>> instanceAssets = Multimaps.synchronizedListMultimap((ListMultimap)ArrayListMultimap.create());
    private final Map<ResourceUrn, ResourceLock> locks = new MapMaker().concurrencyLevel(1).makeMap();
    private final Set<AssetReference<? extends Asset<U>>> references = Sets.newConcurrentHashSet();
    private final ReferenceQueue<Asset<U>> disposalQueue = new ReferenceQueue();
    private volatile boolean closed;
    private volatile ResolutionStrategy resolutionStrategy = (modules, context) -> {
        if (modules.contains(context)) {
            return ImmutableSet.of((Object)context);
        }
        return modules;
    };

    public AssetType(Class<T> assetClass, AssetFactory<T, U> factory) {
        Preconditions.checkNotNull(assetClass);
        Preconditions.checkNotNull(factory);
        this.factory = factory;
        this.assetClass = assetClass;
        Optional assetDataType = GenericsUtil.getTypeParameterBindingForInheritedClass(assetClass, Asset.class, (int)0);
        if (!assetDataType.isPresent()) {
            throw new IllegalArgumentException("Asset class must have a bound AssetData parameter - " + assetClass);
        }
        this.assetDataClass = GenericsUtil.getClassOfType((Type)((Type)assetDataType.get()));
    }

    @Override
    public synchronized void close() {
        if (!this.closed) {
            this.closed = true;
            this.disposeAll();
            this.clearProducers();
        }
    }

    public void processDisposal() {
        Reference<Asset<U>> ref = this.disposalQueue.poll();
        while (ref != null) {
            AssetReference assetRef = (AssetReference)ref;
            assetRef.dispose();
            this.references.remove(assetRef);
            ref = this.disposalQueue.poll();
        }
    }

    public synchronized boolean isClosed() {
        return this.closed;
    }

    public synchronized void disposeAll() {
        this.loadedAssets.values().forEach(Asset::dispose);
        for (WeakReference assetRef : ImmutableList.copyOf((Collection)this.instanceAssets.values())) {
            Asset asset = (Asset)assetRef.get();
            if (asset == null) continue;
            asset.dispose();
        }
        this.processDisposal();
        if (!this.loadedAssets.isEmpty()) {
            logger.error("Assets remained loaded after disposal - {}", this.loadedAssets.keySet());
            this.loadedAssets.clear();
        }
        if (!this.instanceAssets.isEmpty()) {
            logger.error("Asset instances remained loaded after disposal - {}", (Object)this.instanceAssets.keySet());
            this.instanceAssets.clear();
        }
    }

    public void refresh() {
        if (!this.closed) {
            for (Asset asset : this.loadedAssets.values()) {
                if (this.followRedirects(asset.getUrn()).equals(asset.getUrn()) && this.reloadFromProducers(asset)) continue;
                asset.dispose();
                for (WeakReference instanceRef : ImmutableList.copyOf((Collection)this.instanceAssets.get((Object)asset.getUrn().getInstanceUrn()))) {
                    Asset instance = (Asset)instanceRef.get();
                    if (instance == null) continue;
                    instance.dispose();
                }
            }
        }
    }

    public Class<T> getAssetClass() {
        return this.assetClass;
    }

    public Class<U> getAssetDataClass() {
        return this.assetDataClass;
    }

    public void setResolutionStrategy(ResolutionStrategy strategy) {
        this.resolutionStrategy = strategy;
    }

    public synchronized void addProducer(AssetDataProducer<U> producer) {
        if (!this.closed) {
            this.producers.add(producer);
        }
    }

    public List<AssetDataProducer<U>> getProducers() {
        return Collections.unmodifiableList(this.producers);
    }

    public synchronized boolean removeProducer(AssetDataProducer<U> producer) {
        return this.producers.remove(producer);
    }

    public synchronized void clearProducers() {
        this.producers.clear();
    }

    public Optional<T> getAsset(ResourceUrn urn) {
        Preconditions.checkNotNull((Object)urn);
        if (urn.isInstance()) {
            return this.getInstanceAsset(urn);
        }
        return this.getNormalAsset(urn);
    }

    void onAssetDisposed(Asset<U> asset) {
        if (asset.getUrn().isInstance()) {
            this.instanceAssets.get((Object)asset.getUrn()).remove(new WeakReference<T>(this.assetClass.cast(asset)));
        } else {
            this.loadedAssets.remove(asset.getUrn());
        }
    }

    synchronized void registerAsset(Asset<U> asset, DisposalHook disposer) {
        if (this.closed) {
            throw new IllegalStateException("Cannot create asset for disposed asset type: " + this.assetClass);
        }
        if (asset.getUrn().isInstance()) {
            this.instanceAssets.put((Object)asset.getUrn(), new WeakReference<T>(this.assetClass.cast(asset)));
        } else {
            this.loadedAssets.put(asset.getUrn(), this.assetClass.cast(asset));
        }
        this.references.add(new AssetReference<Asset<U>>(asset, this.disposalQueue, disposer));
    }

    public Optional<T> getInstanceAsset(ResourceUrn urn) {
        Optional<T> parentAsset = this.getAsset(urn.getParentUrn());
        if (parentAsset.isPresent()) {
            return this.createInstance((Asset)parentAsset.get());
        }
        return Optional.empty();
    }

    Optional<T> createInstance(Asset<U> asset) {
        Preconditions.checkArgument((boolean)this.assetClass.isAssignableFrom(asset.getClass()));
        Optional<Asset<U>> result = asset.createCopy(asset.getUrn().getInstanceUrn());
        if (!result.isPresent()) {
            try {
                return AccessController.doPrivileged(() -> {
                    for (AssetDataProducer<U> producer : this.producers) {
                        Optional<U> data = producer.getAssetData(asset.getUrn());
                        if (!data.isPresent()) continue;
                        return Optional.of(this.loadAsset(asset.getUrn().getInstanceUrn(), (AssetData)data.get()));
                    }
                    return Optional.ofNullable(this.assetClass.cast(result.get()));
                });
            }
            catch (PrivilegedActionException e) {
                logger.error("Failed to load asset '" + asset.getUrn().getInstanceUrn() + "'", e.getCause());
            }
        }
        return Optional.ofNullable(this.assetClass.cast(result.get()));
    }

    public Optional<T> reload(ResourceUrn urn) {
        Preconditions.checkArgument((!urn.isInstance() ? 1 : 0) != 0, (Object)"Cannot reload an asset instance urn");
        ResourceUrn redirectUrn = this.followRedirects(urn);
        try {
            return AccessController.doPrivileged(() -> {
                for (AssetDataProducer<U> producer : this.producers) {
                    Optional<U> data = producer.getAssetData(redirectUrn);
                    if (!data.isPresent()) continue;
                    return Optional.of(this.loadAsset(redirectUrn, (AssetData)data.get()));
                }
                return Optional.ofNullable(this.loadedAssets.get(redirectUrn));
            });
        }
        catch (PrivilegedActionException e) {
            if (redirectUrn.equals(urn)) {
                logger.error("Failed to load asset '{}'", (Object)redirectUrn, (Object)e.getCause());
            } else {
                logger.error("Failed to load asset '{}' redirected from '{}'", new Object[]{redirectUrn, urn, e.getCause()});
            }
            return Optional.empty();
        }
    }

    private Optional<T> getNormalAsset(ResourceUrn urn) {
        ResourceUrn redirectUrn = this.followRedirects(urn);
        Asset asset = (Asset)this.loadedAssets.get(redirectUrn);
        if (asset == null) {
            return this.reload(redirectUrn);
        }
        return Optional.ofNullable(asset);
    }

    private ResourceUrn followRedirects(ResourceUrn urn) {
        ResourceUrn lastUrn;
        ResourceUrn finalUrn = urn;
        do {
            lastUrn = finalUrn;
            for (AssetDataProducer<U> producer : this.producers) {
                finalUrn = producer.redirect(finalUrn);
            }
        } while (!lastUrn.equals(finalUrn));
        return finalUrn;
    }

    public Optional<T> getAsset(String urn) {
        return this.getAsset(urn, Name.EMPTY);
    }

    public Optional<T> getAsset(String urn, Name moduleContext) {
        Set<ResourceUrn> resolvedUrns = this.resolve(urn, moduleContext);
        if (resolvedUrns.size() == 1) {
            return this.getAsset(resolvedUrns.iterator().next());
        }
        if (resolvedUrns.size() > 1) {
            logger.warn("Failed to resolve asset '{}' - multiple possibilities discovered", (Object)urn);
        } else {
            logger.warn("Failed to resolve asset '{}' - no matches found", (Object)urn);
        }
        return Optional.empty();
    }

    public Set<ResourceUrn> resolve(String urn) {
        return this.resolve(urn, Name.EMPTY);
    }

    public Set<ResourceUrn> resolve(String urn, Name moduleContext) {
        Name fragmentName;
        Name resourceName;
        int fragmentSeparatorIndex;
        if (ResourceUrn.isValid(urn)) {
            return ImmutableSet.of((Object)new ResourceUrn(urn));
        }
        String urnToResolve = urn;
        final boolean instance = urn.endsWith("!instance");
        if (instance) {
            urnToResolve = urn.substring(0, urn.length() - "!instance".length());
        }
        if ((fragmentSeparatorIndex = urnToResolve.indexOf(35)) != -1) {
            resourceName = new Name(urnToResolve.substring(0, fragmentSeparatorIndex));
            fragmentName = new Name(urnToResolve.substring(fragmentSeparatorIndex + 1));
        } else {
            resourceName = new Name(urnToResolve);
            fragmentName = Name.EMPTY;
        }
        Set<Object> possibleModules = Sets.newLinkedHashSet();
        for (AssetDataProducer<U> producer : this.producers) {
            possibleModules.addAll(producer.getModulesProviding(resourceName));
        }
        if (!moduleContext.isEmpty()) {
            possibleModules = this.resolutionStrategy.resolve(possibleModules, moduleContext);
        }
        return Sets.newLinkedHashSet((Iterable)Collections2.transform((Collection)possibleModules, (Function)new Function<Name, ResourceUrn>(){

            @Nullable
            public ResourceUrn apply(Name input) {
                return new ResourceUrn(input, resourceName, fragmentName, instance);
            }
        }));
    }

    private boolean reloadFromProducers(Asset<U> asset) {
        try {
            for (AssetDataProducer<U> producer : this.producers) {
                Optional<U> data = producer.getAssetData(asset.getUrn());
                if (!data.isPresent()) continue;
                asset.reload((AssetData)data.get());
                for (WeakReference assetInstanceRef : this.instanceAssets.get((Object)asset.getUrn().getInstanceUrn())) {
                    Asset assetInstance = (Asset)assetInstanceRef.get();
                    if (assetInstance == null) continue;
                    assetInstance.reload((AssetData)data.get());
                }
                return true;
            }
        }
        catch (IOException e) {
            logger.error("Failed to reload asset '{}', disposing", (Object)asset.getUrn());
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public T loadAsset(ResourceUrn urn, U data) {
        if (urn.isInstance()) {
            return this.factory.build(urn, this, data);
        }
        Asset asset = (Asset)this.loadedAssets.get(urn);
        if (asset != null) {
            asset.reload(data);
        } else {
            ResourceLock lock;
            Map<ResourceUrn, ResourceLock> map = this.locks;
            synchronized (map) {
                lock = this.locks.computeIfAbsent(urn, k -> new ResourceLock(urn));
            }
            try {
                lock.lock();
                if (!this.closed) {
                    asset = (Asset)this.loadedAssets.get(urn);
                    if (asset == null) {
                        asset = this.factory.build(urn, this, data);
                    } else {
                        asset.reload(data);
                    }
                }
                map = this.locks;
                synchronized (map) {
                    if (lock.unlock()) {
                        this.locks.remove(urn);
                    }
                }
            }
            catch (InterruptedException e) {
                logger.error("Failed to load asset - interrupted awaiting lock on resource {}", (Object)urn);
            }
        }
        return (T)asset;
    }

    public boolean isLoaded(ResourceUrn urn) {
        Preconditions.checkArgument((!urn.isInstance() ? 1 : 0) != 0, (Object)"Urn must not be an instance urn");
        return this.loadedAssets.containsKey(urn);
    }

    public Set<ResourceUrn> getLoadedAssetUrns() {
        return ImmutableSet.copyOf(this.loadedAssets.keySet());
    }

    public Set<T> getLoadedAssets() {
        return ImmutableSet.copyOf(this.loadedAssets.values());
    }

    public Set<ResourceUrn> getAvailableAssetUrns() {
        LinkedHashSet availableAssets = Sets.newLinkedHashSet(this.getLoadedAssetUrns());
        for (AssetDataProducer<U> producer : this.producers) {
            availableAssets.addAll(producer.getAvailableAssetUrns());
        }
        return availableAssets;
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof AssetType) {
            AssetType other = (AssetType)obj;
            return this.assetClass.equals(other.assetClass);
        }
        return false;
    }

    public int hashCode() {
        return this.assetClass.hashCode();
    }

    public String toString() {
        return this.assetClass.getSimpleName();
    }

    private static final class AssetReference<T>
    extends PhantomReference<T> {
        private final DisposalHook disposalHook;

        public AssetReference(T asset, ReferenceQueue<T> queue, DisposalHook hook) {
            super(asset, queue);
            this.disposalHook = hook;
        }

        public void dispose() {
            this.disposalHook.dispose();
        }
    }

    private static final class ResourceLock {
        private final ResourceUrn urn;
        private final Semaphore semaphore = new Semaphore(1);

        public ResourceLock(ResourceUrn urn) {
            this.urn = urn;
        }

        public void lock() throws InterruptedException {
            this.semaphore.acquire();
        }

        public boolean unlock() {
            boolean lockFinished = !this.semaphore.hasQueuedThreads();
            this.semaphore.release();
            return lockFinished;
        }

        public String toString() {
            return "lock(" + this.urn + ")";
        }
    }
}

