/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.worldwind.terrain;

import gov.nasa.worldwind.WWObjectImpl;
import gov.nasa.worldwind.cache.BasicMemoryCache;
import gov.nasa.worldwind.cache.MemoryCache;
import gov.nasa.worldwind.exception.WWRuntimeException;
import gov.nasa.worldwind.exception.WWTimeoutException;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Box;
import gov.nasa.worldwind.geom.Extent;
import gov.nasa.worldwind.geom.Intersection;
import gov.nasa.worldwind.geom.LatLon;
import gov.nasa.worldwind.geom.Line;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.geom.Triangle;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.globes.ElevationModel;
import gov.nasa.worldwind.globes.Globe;
import gov.nasa.worldwind.render.SurfaceQuad;
import gov.nasa.worldwind.terrain.Terrain;
import gov.nasa.worldwind.util.Logging;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public class HighResolutionTerrain
extends WWObjectImpl
implements Terrain {
    protected static final int DEFAULT_DENSITY = 3;
    protected static final long DEFAULT_CACHE_CAPACITY = 200000000L;
    protected Globe globe;
    protected Sector sector;
    protected double verticalExaggeration = 1.0;
    protected Long timeout;
    protected int density = 3;
    protected double targetResolution;
    protected double latTileSize;
    protected double lonTileSize;
    protected int numRows;
    protected int numCols;
    protected MemoryCache geometryCache;
    protected ThreadLocal<Long> startTime = new ThreadLocal();
    protected boolean useCachedElevationsOnly = false;

    public HighResolutionTerrain(Globe globe, Double d) {
        this(globe, null, d, null);
    }

    public HighResolutionTerrain(Globe globe, Sector sector, Double d, Double d2) {
        if (globe == null) {
            String string = Logging.getMessage("nullValue.GlobeIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        this.globe = globe;
        this.sector = sector != null ? sector : Sector.FULL_SPHERE;
        this.targetResolution = d != null ? d / this.globe.getRadius() : this.globe.getElevationModel().getBestResolution(null);
        this.verticalExaggeration = d2 != null ? d2 : 1.0;
        this.computeDimensions();
        this.geometryCache = new BasicMemoryCache(170000000L, 200000000L);
    }

    public double getCacheUsage() {
        return (double)this.geometryCache.getUsedCapacity() / (double)this.geometryCache.getCapacity();
    }

    public int getNumCacheEntries() {
        return this.geometryCache.getNumObjects();
    }

    @Override
    public Globe getGlobe() {
        return this.globe;
    }

    public Sector getSector() {
        return this.sector;
    }

    public double getTargetResolution() {
        return this.targetResolution;
    }

    @Override
    public double getVerticalExaggeration() {
        return this.verticalExaggeration;
    }

    public synchronized Long getTimeout() {
        return this.timeout;
    }

    public synchronized void setTimeout(Long l) {
        this.timeout = l;
    }

    public int getDensity() {
        return this.density;
    }

    public void setDensity(int n) {
        this.density = n;
        this.computeDimensions();
    }

    public long getCacheCapacity() {
        return this.geometryCache.getCapacity();
    }

    public void setCacheCapacity(long l) {
        this.geometryCache.setCapacity(Math.max(l, 1000000L));
    }

    @Override
    public Vec4 getSurfacePoint(Position position) {
        return this.getSurfacePoint(position.getLatitude(), position.getLongitude(), position.getAltitude());
    }

    @Override
    public Vec4 getSurfacePoint(Angle angle, Angle angle2, double d) {
        if (angle == null || angle2 == null) {
            String string = Logging.getMessage("nullValue.LatLonIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        try {
            this.startTime.set(System.currentTimeMillis());
            RectTile rectTile = this.getContainingTile(angle, angle2);
            Vec4 vec4 = rectTile != null ? this.getSurfacePoint(rectTile, angle, angle2, d) : null;
            return vec4;
        }
        catch (InterruptedException interruptedException) {
            throw new WWRuntimeException(interruptedException);
        }
        finally {
            this.startTime.set(null);
        }
    }

    @Override
    public Double getElevation(LatLon latLon) {
        if (latLon == null) {
            String string = Logging.getMessage("nullValue.LatLonIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        Vec4 vec4 = this.getSurfacePoint(latLon.getLatitude(), latLon.getLongitude(), 0.0);
        if (vec4 == null) {
            return null;
        }
        Vec4 vec42 = this.globe.computePointFromPosition(latLon.getLatitude(), latLon.getLongitude(), 0.0);
        return vec42.distanceTo3(vec4) * (double)(vec4.getLength3() >= vec42.getLength3() ? 1 : -1);
    }

    public Intersection[] intersect(Line line) {
        if (line == null) {
            String string = Logging.getMessage("nullValue.LineIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        Position position = this.globe.computePositionFromPoint(line.getOrigin());
        Intersection[] intersectionArray = this.globe.intersect(line, 0.0);
        if (intersectionArray == null || intersectionArray.length == 0) {
            return null;
        }
        Position position2 = this.globe.computePositionFromPoint(intersectionArray[0].getIntersectionPoint());
        return this.intersect(position, position2);
    }

    @Override
    public Intersection[] intersect(Position position, Position position2) {
        if (position == null || position2 == null) {
            String string = Logging.getMessage("nullValue.PositionIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        try {
            this.startTime.set(System.currentTimeMillis());
            Intersection[] intersectionArray = this.doIntersect(position, position2);
            return intersectionArray;
        }
        catch (InterruptedException interruptedException) {
            throw new WWRuntimeException(interruptedException);
        }
        finally {
            this.startTime.set(null);
        }
    }

    @Override
    public Intersection[] intersect(Position position, Position position2, int n) {
        if (position == null || position2 == null) {
            String string = Logging.getMessage("nullValue.PositionIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        double d = position.getAltitude();
        double d2 = position2.getAltitude();
        if (n == 0) {
            d -= this.getElevation(position).doubleValue();
            d2 -= this.getElevation(position2).doubleValue();
        } else if (n == 1) {
            d = 0.0;
            d2 = 0.0;
        }
        return this.intersect(new Position(position, d), new Position(position2, d2));
    }

    public void intersect(List<Position> list, final IntersectionCallback intersectionCallback) throws InterruptedException {
        ExecutorService executorService = Executors.newFixedThreadPool(10);
        for (int i = 0; i < list.size(); i += 2) {
            final Position position = list.get(i);
            final Position position2 = list.get(i + 1);
            executorService.submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        Intersection[] intersectionArray = HighResolutionTerrain.this.intersect(position, position2);
                        if (intersectionArray != null) {
                            intersectionCallback.intersection(position, position2, intersectionArray);
                        }
                    }
                    catch (Exception exception) {
                        intersectionCallback.exception(exception);
                    }
                }
            });
        }
        executorService.shutdown();
        executorService.awaitTermination(100L, TimeUnit.DAYS);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cacheIntersectingTiles(Position position, Position position2) throws InterruptedException {
        if (position == null || position2 == null) {
            String string = Logging.getMessage("nullValue.PositionIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        Line line = this.makeLineFromPositions(position, position2);
        if (line == null) {
            return;
        }
        try {
            this.startTime.set(System.currentTimeMillis());
            List<RectTile> list = this.getIntersectingTiles(position, position2, line);
            if (list == null) {
                return;
            }
            for (RectTile rectTile : list) {
                if (rectTile.ri != null) continue;
                this.makeVerts(rectTile);
            }
        }
        finally {
            this.startTime.set(null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cacheIntersectingTiles(Sector sector) throws InterruptedException {
        if (sector == null) {
            String string = Logging.getMessage("nullValue.SectorIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        try {
            this.startTime.set(System.currentTimeMillis());
            List<RectTile> list = this.getIntersectingTiles(sector);
            if (list == null) {
                return;
            }
            for (RectTile rectTile : list) {
                this.makeVerts(rectTile);
            }
        }
        finally {
            this.startTime.set(null);
        }
    }

    public List<Sector> getIntersectionTiles(Position position, Position position2) throws InterruptedException {
        Line line = this.makeLineFromPositions(position, position2);
        if (line == null) {
            return null;
        }
        List<RectTile> list = this.getIntersectingTiles(position, position2, line);
        if (list == null || list.size() == 0) {
            return null;
        }
        ArrayList<Sector> arrayList = new ArrayList<Sector>(list.size());
        for (RectTile rectTile : list) {
            arrayList.add(rectTile.sector);
        }
        return arrayList;
    }

    protected void computeDimensions() {
        double d = Math.max(this.globe.getElevationModel().getBestResolution(null), this.targetResolution);
        this.numCols = (int)Math.ceil(this.sector.getDeltaLonRadians() / ((double)this.density * d));
        this.numRows = (int)Math.ceil(this.sector.getDeltaLatRadians() / ((double)this.density * d));
        this.lonTileSize = this.sector.getDeltaLonDegrees() / (double)(this.numCols - 1);
        this.latTileSize = this.sector.getDeltaLatDegrees() / (double)(this.numRows - 1);
        if (this.geometryCache != null) {
            this.geometryCache.clear();
        }
    }

    protected RectTile getContainingTile(Angle angle, Angle angle2) {
        if (!this.sector.contains(angle, angle2)) {
            return null;
        }
        int n = this.computeRow(this.sector, angle);
        int n2 = this.computeColumn(this.sector, angle2);
        return this.createTile(n, n2);
    }

    protected RectTile createTile(int n, int n2) {
        if (n < 0 || n2 < 0 || n >= this.numRows || n2 >= this.numCols) {
            return null;
        }
        double d = Math.max(this.sector.getMinLongitude().degrees + (double)n2 * this.lonTileSize, -180.0);
        double d2 = Math.min(d + this.lonTileSize, 180.0);
        double d3 = Math.max(this.sector.getMinLatitude().degrees + (double)n * this.latTileSize, -90.0);
        double d4 = Math.min(d3 + this.latTileSize, 90.0);
        return this.createTile(Sector.fromDegrees(d3, d4, d, d2));
    }

    protected RectTile createTile(Sector sector) {
        Box box = Sector.computeBoundingBox(this.globe, this.verticalExaggeration, sector);
        return new RectTile(box, this.density, sector);
    }

    protected int computeRow(Sector sector, Angle angle) {
        double d = sector.getMaxLatitude().degrees;
        double d2 = sector.getMinLatitude().degrees;
        double d3 = (angle.degrees - d2) / (d - d2);
        return (int)(d3 * (double)(this.numRows - 1));
    }

    protected int computeColumn(Sector sector, Angle angle) {
        double d = sector.getMaxLongitude().degrees;
        double d2 = sector.getMinLongitude().degrees;
        double d3 = (angle.degrees - d2) / (d - d2);
        return (int)(d3 * (double)(this.numCols - 1));
    }

    protected Line makeLineFromPositions(Position position, Position position2) throws InterruptedException {
        if (position == null || position2 == null) {
            String string = Logging.getMessage("nullValue.PositionIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        RectTile rectTile = this.getContainingTile(position.getLatitude(), position.getLongitude());
        RectTile rectTile2 = this.getContainingTile(position2.getLatitude(), position2.getLongitude());
        if (rectTile == null || rectTile2 == null) {
            return null;
        }
        Vec4 vec4 = this.getSurfacePoint(rectTile, position.getLatitude(), position.getLongitude(), position.getAltitude());
        Vec4 vec42 = this.getSurfacePoint(rectTile2, position2.getLatitude(), position2.getLongitude(), position2.getAltitude());
        if (vec4 == null || vec42 == null) {
            return null;
        }
        if (position.getLatitude().equals(position2.getLatitude()) && position.getLongitude().equals(position2.getLongitude()) && position.getAltitude() == position2.getAltitude()) {
            return null;
        }
        return new Line(vec4, vec42.subtract3(vec4));
    }

    protected Intersection[] doIntersect(Position position, Position position2) throws InterruptedException {
        Intersection[] intersectionArray;
        Line line = this.makeLineFromPositions(position, position2);
        if (line == null) {
            return null;
        }
        List<RectTile> list = this.getIntersectingTiles(position, position2, line);
        if (list == null) {
            return null;
        }
        ArrayList<Intersection> arrayList = new ArrayList<Intersection>();
        for (RectTile rectTile : list) {
            intersectionArray = this.intersect(rectTile, line);
            if (intersectionArray == null) continue;
            arrayList.addAll(Arrays.asList(intersectionArray));
        }
        if (arrayList.size() == 0) {
            return null;
        }
        intersectionArray = new Intersection[arrayList.size()];
        arrayList.toArray(intersectionArray);
        if (arrayList.size() == 1) {
            return intersectionArray;
        }
        final Vec4 vec4 = line.getOrigin();
        Arrays.sort(intersectionArray, new Comparator<Intersection>(){

            @Override
            public int compare(Intersection intersection, Intersection intersection2) {
                if (intersection == null && intersection2 == null) {
                    return 0;
                }
                if (intersection2 == null) {
                    return -1;
                }
                if (intersection == null) {
                    return 1;
                }
                Vec4 vec43 = intersection.getIntersectionPoint();
                Vec4 vec42 = intersection2.getIntersectionPoint();
                double d = vec4.distanceTo3(vec43);
                double d2 = vec4.distanceTo3(vec42);
                return Double.compare(d, d2);
            }
        });
        return intersectionArray;
    }

    protected List<RectTile> getIntersectingTiles(Sector sector) {
        int n = this.computeRow(this.sector, sector.getMinLatitude());
        int n2 = this.computeColumn(this.sector, sector.getMinLongitude());
        int n3 = this.computeRow(this.sector, sector.getMaxLatitude());
        int n4 = this.computeColumn(this.sector, sector.getMaxLongitude());
        int n5 = (1 + (n3 - n)) * (1 + (n4 - n2));
        ArrayList<RectTile> arrayList = new ArrayList<RectTile>(n5);
        for (int i = n2; i <= n4; ++i) {
            for (int j = n; j <= n3; ++j) {
                arrayList.add(this.createTile(j, i));
            }
        }
        return arrayList;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected List<RectTile> getIntersectingTiles(Position position, Position position2, Line line) {
        boolean bl = this.getGlobe().getElevationModel().isExtremesCachingEnabled();
        this.getGlobe().getElevationModel().setExtremesCachingEnabled(false);
        try {
            int n;
            int n2 = this.computeRow(this.sector, position.getLatitude());
            int n3 = this.computeColumn(this.sector, position.getLongitude());
            int n4 = this.computeRow(this.sector, position2.getLatitude());
            int n5 = this.computeColumn(this.sector, position2.getLongitude());
            if (n4 < n2) {
                n = n2;
                n2 = n4;
                n4 = n;
            }
            if (n5 < n3) {
                n = n3;
                n3 = n5;
                n5 = n;
            }
            ArrayList<RectTile> arrayList = new ArrayList<RectTile>();
            this.doGetIntersectingTiles(n2, n3, n4, n5, line, arrayList);
            ArrayList<RectTile> arrayList2 = arrayList.size() > 0 ? arrayList : null;
            return arrayList2;
        }
        finally {
            this.getGlobe().getElevationModel().setExtremesCachingEnabled(bl);
        }
    }

    protected void doGetIntersectingTiles(int n, int n2, int n3, int n4, Line line, List<RectTile> list) {
        double d = this.sector.getMinLatitude().degrees + (double)n * this.latTileSize;
        double d2 = this.sector.getMinLatitude().degrees + (double)(n3 + 1) * this.latTileSize;
        double d3 = this.sector.getMinLongitude().degrees + (double)n2 * this.lonTileSize;
        double d4 = this.sector.getMinLongitude().degrees + (double)(n4 + 1) * this.lonTileSize;
        Box box = Sector.computeBoundingBox(this.globe, this.verticalExaggeration, Sector.fromDegrees(d, d2, d3, d4));
        if (!box.intersects(line)) {
            return;
        }
        int n5 = n4 - n2 + 1;
        int n6 = n3 - n + 1;
        if (n5 == 1 && n6 == 1) {
            list.add(this.createTile(n, n2));
            return;
        }
        this.doGetIntersectingTiles(n, n2, n + Math.max(0, n6 / 2 - 1), n2 + Math.max(0, n5 / 2 - 1), line, list);
        if (n5 != 1) {
            this.doGetIntersectingTiles(n, n2 + n5 / 2, n + Math.max(0, n6 / 2 - 1), n4, line, list);
        }
        if (n6 != 1) {
            this.doGetIntersectingTiles(n + n6 / 2, n2, n3, n2 + Math.max(0, n5 / 2 - 1), line, list);
        }
        if (n5 != 1 && n6 != 1) {
            this.doGetIntersectingTiles(n + n6 / 2, n2 + n5 / 2, n3, n4, line, list);
        }
    }

    protected void makeVerts(RectTile rectTile) throws InterruptedException {
        rectTile.ri = (RenderInfo)this.geometryCache.getObject(rectTile.sector);
        if (rectTile.ri != null) {
            return;
        }
        rectTile.ri = this.buildVerts(rectTile);
        if (rectTile.ri != null) {
            this.geometryCache.add(rectTile.sector, rectTile.ri, rectTile.ri.getSizeInBytes());
        }
    }

    protected RenderInfo buildVerts(RectTile rectTile) throws InterruptedException {
        int n = rectTile.density;
        int n2 = (n + 1) * (n + 1);
        float[] fArray = rectTile.ri == null || rectTile.ri.vertices == null ? new float[n2 * 3] : rectTile.ri.vertices;
        ArrayList<LatLon> arrayList = this.computeLocations(rectTile);
        double[] dArray = new double[arrayList.size()];
        double[] dArray2 = this.getGlobe().getElevationModel().getBestResolutions(this.sector);
        for (int i = 0; i < dArray2.length; ++i) {
            dArray2[i] = Math.max(dArray2[i], this.targetResolution);
        }
        this.getElevations(rectTile.sector, arrayList, dArray2, dArray);
        LatLon latLon = rectTile.sector.getCentroid();
        Vec4 vec4 = this.globe.computePointFromPosition(latLon.getLatitude(), latLon.getLongitude(), 0.0);
        double d = Double.MAX_VALUE;
        double d2 = -1.7976931348623157E308;
        LatLon latLon2 = latLon;
        LatLon latLon3 = latLon;
        int n3 = 0;
        int n4 = 0;
        Iterator<LatLon> iterator = arrayList.iterator();
        for (int i = 0; i <= n; ++i) {
            for (int j = 0; j <= n; ++j) {
                LatLon latLon4 = iterator.next();
                int n5 = n3++;
                double d3 = this.verticalExaggeration * dArray[n5];
                if (d3 < d) {
                    d = d3;
                    latLon2 = latLon4;
                }
                if (d3 > d2) {
                    d2 = d3;
                    latLon3 = latLon4;
                }
                Vec4 vec42 = this.globe.computePointFromPosition(latLon4.getLatitude(), latLon4.getLongitude(), d3);
                fArray[n4++] = (float)(vec42.x - vec4.x);
                fArray[n4++] = (float)(vec42.y - vec4.y);
                fArray[n4++] = (float)(vec42.z - vec4.z);
            }
        }
        return new RenderInfo(n, fArray, vec4, new Position(latLon2, d), new Position(latLon3, d2));
    }

    public void setUseCachedElevationsOnly(boolean bl) {
        this.useCachedElevationsOnly = bl;
    }

    public boolean isUseCachedElevationsOnly() {
        return this.useCachedElevationsOnly;
    }

    protected void getElevations(Sector sector, List<LatLon> list, double[] dArray, double[] dArray2) throws InterruptedException {
        if (this.useCachedElevationsOnly) {
            this.getCachedElevations(list, dArray2);
            return;
        }
        double[] dArray3 = new double[dArray.length];
        for (int i = 0; i < dArray.length; ++i) {
            dArray3[i] = Double.MAX_VALUE;
        }
        while (!this.resolutionsMeetCriteria(dArray3, dArray) && !this.resolutionsMeetCriteria(dArray3 = this.globe.getElevations(sector, list, dArray, dArray2), dArray)) {
            Thread.sleep(this.timeout == null ? 5L : Math.min(this.timeout, 5L));
            Long l = this.getTimeout();
            if (this.startTime.get() == null || l == null || System.currentTimeMillis() - this.startTime.get() <= l) continue;
            throw new WWTimeoutException("Terrain convergence timed out");
        }
    }

    protected boolean resolutionsMeetCriteria(double[] dArray, double[] dArray2) {
        for (int i = 0; i < dArray.length; ++i) {
            if (!(dArray[i] > dArray2[i])) continue;
            return false;
        }
        return true;
    }

    protected void getCachedElevations(List<LatLon> list, double[] dArray) {
        ElevationModel elevationModel = this.globe.getElevationModel();
        for (int i = 0; i < list.size(); ++i) {
            LatLon latLon = list.get(i);
            double d = elevationModel.getUnmappedLocalSourceElevation(latLon.latitude, latLon.longitude);
            if (d == elevationModel.getMissingDataSignal()) {
                d = elevationModel.getMissingDataReplacement();
            }
            dArray[i] = d;
        }
    }

    protected ArrayList<LatLon> computeLocations(RectTile rectTile) {
        int n = rectTile.density;
        int n2 = (n + 1) * (n + 1);
        Angle angle = rectTile.sector.getMaxLatitude();
        Angle angle2 = rectTile.sector.getDeltaLat().divide(n);
        Angle angle3 = rectTile.sector.getMinLatitude();
        Angle angle4 = rectTile.sector.getMinLongitude();
        Angle angle5 = rectTile.sector.getMaxLongitude();
        Angle angle6 = rectTile.sector.getDeltaLon().divide(n);
        ArrayList<LatLon> arrayList = new ArrayList<LatLon>(n2);
        for (int i = 0; i <= n; ++i) {
            Angle angle7 = angle4;
            for (int j = 0; j <= n; ++j) {
                arrayList.add(new LatLon(angle3, angle7));
                angle7 = j == n ? angle5 : angle7.add(angle6);
                if (angle7.degrees < -180.0) {
                    angle7 = Angle.NEG180;
                    continue;
                }
                if (!(angle7.degrees > 180.0)) continue;
                angle7 = Angle.POS180;
            }
            angle3 = i == n ? angle : angle3.add(angle2);
        }
        return arrayList;
    }

    protected Vec4 getSurfacePoint(RectTile rectTile, Angle angle, Angle angle2, double d) throws InterruptedException {
        Vec4 vec4 = this.getSurfacePoint(rectTile, angle, angle2);
        if (d != 0.0 && vec4 != null) {
            vec4 = this.applyOffset(vec4, d);
        }
        return vec4;
    }

    protected Vec4 applyOffset(Vec4 vec4, double d) {
        Vec4 vec42 = this.globe.computeSurfaceNormalAtPoint(vec4);
        vec4 = Vec4.fromLine3(vec4, d, vec42);
        return vec4;
    }

    protected Vec4 getSurfacePoint(RectTile rectTile, Angle angle, Angle angle2) throws InterruptedException {
        if (angle == null || angle2 == null) {
            String string = Logging.getMessage("nullValue.LatLonIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (!rectTile.sector.contains(angle, angle2)) {
            return null;
        }
        if (rectTile.ri == null) {
            this.makeVerts(rectTile);
        }
        if (rectTile.ri == null) {
            return null;
        }
        double d = angle.getDegrees();
        double d2 = angle2.getDegrees();
        double d3 = rectTile.sector.getMinLatitude().getDegrees();
        double d4 = rectTile.sector.getMaxLatitude().getDegrees();
        double d5 = rectTile.sector.getMinLongitude().getDegrees();
        double d6 = rectTile.sector.getMaxLongitude().getDegrees();
        double d7 = (d2 - d5) / (d6 - d5);
        double d8 = (d - d3) / (d4 - d3);
        int n = (int)(d8 * (double)rectTile.density);
        int n2 = (int)(d7 * (double)rectTile.density);
        double d9 = HighResolutionTerrain.createPosition(n2, d7, rectTile.ri.density);
        double d10 = HighResolutionTerrain.createPosition(n, d8, rectTile.ri.density);
        Vec4 vec4 = HighResolutionTerrain.interpolate(n, n2, d9, d10, rectTile.ri);
        vec4 = vec4.add3(rectTile.ri.referenceCenter);
        return vec4;
    }

    protected static double createPosition(int n, double d, int n2) {
        double d2 = (double)n / (double)n2;
        double d3 = (double)(n + 1) / (double)n2;
        return (d - d2) / (d3 - d2);
    }

    protected static Vec4 interpolate(int n, int n2, double d, double d2, RenderInfo renderInfo) {
        int n3 = renderInfo.density + 1;
        int n4 = n * n3 + n2;
        int n5 = n3 * 3;
        int n6 = n4 *= 3;
        Vec4 vec4 = new Vec4(renderInfo.vertices[n6++], renderInfo.vertices[n6++], renderInfo.vertices[n6++]);
        Vec4 vec42 = new Vec4(renderInfo.vertices[n6++], renderInfo.vertices[n6++], renderInfo.vertices[n6]);
        n6 = n4 += n5;
        Vec4 vec43 = new Vec4(renderInfo.vertices[n6++], renderInfo.vertices[n6++], renderInfo.vertices[n6++]);
        Vec4 vec44 = new Vec4(renderInfo.vertices[n6++], renderInfo.vertices[n6++], renderInfo.vertices[n6]);
        return HighResolutionTerrain.interpolate(vec4, vec42, vec44, vec43, d, d2);
    }

    protected static Vec4 interpolate(Vec4 vec4, Vec4 vec42, Vec4 vec43, Vec4 vec44, double d, double d2) {
        double d3 = d + d2;
        if (d3 == 1.0) {
            return new Vec4(vec44.x * d2 + vec42.x * d, vec44.y * d2 + vec42.y * d, vec44.z * d2 + vec42.z * d);
        }
        if (d3 > 1.0) {
            Vec4 vec45 = vec44.subtract3(vec43).multiply3(1.0 - d);
            Vec4 vec46 = vec42.subtract3(vec43).multiply3(1.0 - d2);
            return vec43.add3(vec45).add3(vec46);
        }
        Vec4 vec47 = vec42.subtract3(vec4).multiply3(d);
        Vec4 vec48 = vec44.subtract3(vec4).multiply3(d2);
        return vec4.add3(vec47).add3(vec48);
    }

    protected Intersection[] intersect(RectTile rectTile, Line line) throws InterruptedException {
        int n;
        if (rectTile.ri == null) {
            this.makeVerts(rectTile);
        }
        if (rectTile.ri == null) {
            return null;
        }
        ArrayList<Intersection> arrayList = new ArrayList<Intersection>();
        double d = rectTile.ri.referenceCenter.x;
        double d2 = rectTile.ri.referenceCenter.y;
        double d3 = rectTile.ri.referenceCenter.z;
        int n2 = rectTile.density + 1;
        float[] fArray = rectTile.ri.vertices;
        for (n = 0; n < n2 - 1; ++n) {
            for (int i = 0; i < n2 - 1; ++i) {
                int n3 = (n * n2 + i) * 3;
                Vec4 vec4 = new Vec4((double)fArray[n3] + d, (double)fArray[n3 + 1] + d2, (double)fArray[n3 + 2] + d3);
                Vec4 vec42 = new Vec4((double)fArray[n3 += 3] + d, (double)fArray[n3 + 1] + d2, (double)fArray[n3 + 2] + d3);
                Vec4 vec43 = new Vec4((double)fArray[n3 += n2 * 3] + d, (double)fArray[n3 + 1] + d2, (double)fArray[n3 + 2] + d3);
                Vec4 vec44 = new Vec4((double)fArray[n3 -= 3] + d, (double)fArray[n3 + 1] + d2, (double)fArray[n3 + 2] + d3);
                Intersection intersection = Triangle.intersect(line, vec4, vec42, vec43);
                if (intersection != null) {
                    arrayList.add(intersection);
                }
                if ((intersection = Triangle.intersect(line, vec4, vec43, vec44)) == null) continue;
                arrayList.add(intersection);
            }
        }
        n = arrayList.size();
        if (n == 0) {
            return null;
        }
        Intersection[] intersectionArray = new Intersection[n];
        arrayList.toArray(intersectionArray);
        final Vec4 vec4 = line.getOrigin();
        Arrays.sort(intersectionArray, new Comparator<Intersection>(){

            @Override
            public int compare(Intersection intersection, Intersection intersection2) {
                if (intersection == null && intersection2 == null) {
                    return 0;
                }
                if (intersection2 == null) {
                    return -1;
                }
                if (intersection == null) {
                    return 1;
                }
                Vec4 vec43 = intersection.getIntersectionPoint();
                Vec4 vec42 = intersection2.getIntersectionPoint();
                double d = vec4.distanceTo3(vec43);
                double d2 = vec4.distanceTo3(vec42);
                return Double.compare(d, d2);
            }
        });
        return intersectionArray;
    }

    protected List<Vec4[]> intersect(RectTile rectTile, Vec4[] vec4Array) throws InterruptedException {
        int n;
        if (rectTile.ri == null) {
            this.makeVerts(rectTile);
        }
        if (rectTile.ri == null) {
            return null;
        }
        ArrayList<Vec4[]> arrayList = new ArrayList<Vec4[]>();
        double d = rectTile.ri.referenceCenter.x;
        double d2 = rectTile.ri.referenceCenter.y;
        double d3 = rectTile.ri.referenceCenter.z;
        int n2 = rectTile.density + 1;
        float[] fArray = rectTile.ri.vertices;
        Vec4[] vec4Array2 = new Vec4[3];
        Vec4[] vec4Array3 = new Vec4[3];
        Vec4[] vec4Array4 = new Vec4[3];
        for (n = 0; n < n2 - 1; ++n) {
            for (int i = 0; i < n2 - 1; ++i) {
                int n3 = (n * n2 + i) * 3;
                vec4Array2[0] = new Vec4((double)fArray[n3] + d, (double)fArray[n3 + 1] + d2, (double)fArray[n3 + 2] + d3);
                vec4Array3[0] = vec4Array2[0];
                vec4Array2[1] = new Vec4((double)fArray[n3 += 3] + d, (double)fArray[n3 + 1] + d2, (double)fArray[n3 + 2] + d3);
                vec4Array2[2] = new Vec4((double)fArray[n3 += n2 * 3] + d, (double)fArray[n3 + 1] + d2, (double)fArray[n3 + 2] + d3);
                vec4Array3[1] = vec4Array2[2];
                vec4Array3[2] = new Vec4((double)fArray[n3 -= 3] + d, (double)fArray[n3 + 1] + d2, (double)fArray[n3 + 2] + d3);
                int n4 = Triangle.intersectTriangles(vec4Array, vec4Array2, vec4Array4);
                if (n4 == 1) {
                    arrayList.add(new Vec4[]{vec4Array4[0], vec4Array4[1]});
                } else if (n4 == 0) {
                    arrayList.add(new Vec4[]{vec4Array2[0], vec4Array2[1], vec4Array2[2]});
                }
                n4 = Triangle.intersectTriangles(vec4Array, vec4Array3, vec4Array4);
                if (n4 == 1) {
                    arrayList.add(new Vec4[]{vec4Array4[0], vec4Array4[1]});
                    continue;
                }
                if (n4 != 0) continue;
                arrayList.add(new Vec4[]{vec4Array3[0], vec4Array3[1], vec4Array3[2]});
            }
        }
        n = arrayList.size();
        if (n == 0) {
            return null;
        }
        return arrayList;
    }

    public void intersectTriangle(Vec4[] vec4Array, Position[] positionArray, List<Position[]> list) throws InterruptedException {
        Sector sector = Sector.boundingSector(Arrays.asList(positionArray));
        List<RectTile> list2 = this.getIntersectingTiles(sector);
        double d = positionArray[0].getAltitude();
        for (int i = 1; i < positionArray.length; ++i) {
            if (!(positionArray[i].getAltitude() < d)) continue;
            d = positionArray[i].getAltitude();
        }
        list2 = this.eliminateLowAltitudeTiles(list2, d);
        ArrayList<Vec4[]> arrayList = new ArrayList<Vec4[]>();
        for (RectTile rectTile : list2) {
            List<Vec4[]> list3 = this.intersect(rectTile, vec4Array);
            if (list3 == null) continue;
            arrayList.addAll(list3);
        }
        this.convertPointsToPositions(arrayList, list);
    }

    protected List<RectTile> eliminateLowAltitudeTiles(List<RectTile> list, double d) throws InterruptedException {
        ArrayList<RectTile> arrayList = new ArrayList<RectTile>();
        for (RectTile rectTile : list) {
            if (rectTile.ri == null) {
                this.makeVerts(rectTile);
            }
            if (rectTile.ri == null) {
                return null;
            }
            if (!(rectTile.ri.maxElevation.getElevation() >= d)) continue;
            arrayList.add(rectTile);
        }
        return arrayList;
    }

    protected void convertPointsToPositions(List<Vec4[]> list, List<Position[]> list2) {
        for (Vec4[] vec4Array : list) {
            Position[] positionArray = new Position[vec4Array.length];
            for (int i = 0; i < vec4Array.length; ++i) {
                positionArray[i] = this.getGlobe().computePositionFromPoint(vec4Array[i]);
            }
            list2.add(positionArray);
        }
    }

    public Position[] getExtremeElevations(Sector sector) throws InterruptedException {
        List<RectTile> list = this.getIntersectingTiles(sector);
        this.startTime.set(System.currentTimeMillis());
        Position[] positionArray = new Position[2];
        for (RectTile rectTile : list) {
            if (rectTile.ri == null) {
                this.makeVerts(rectTile);
            }
            if (rectTile.ri == null) continue;
            if (positionArray[0] == null || rectTile.ri.minElevation.getElevation() < positionArray[0].getElevation()) {
                positionArray[0] = rectTile.ri.minElevation;
            }
            if (positionArray[1] != null && !(rectTile.ri.maxElevation.getElevation() > positionArray[1].getElevation())) continue;
            positionArray[1] = rectTile.ri.maxElevation;
        }
        return positionArray;
    }

    public Position[] getExtremeElevations(LatLon latLon, double d, double d2) throws InterruptedException {
        SurfaceQuad surfaceQuad = new SurfaceQuad(latLon, d, d2);
        Sector sector = Sector.boundingSector(surfaceQuad.getLocations(this.getGlobe()));
        return this.getExtremeElevations(sector);
    }

    protected static class RectTile {
        protected final Sector sector;
        protected final int density;
        protected Extent extent;
        protected RenderInfo ri;

        public RectTile(Extent extent, int n, Sector sector) {
            this.density = n;
            this.sector = sector;
            this.extent = extent;
        }
    }

    public static interface IntersectionCallback {
        public void intersection(Position var1, Position var2, Intersection[] var3);

        public void exception(Exception var1);
    }

    protected static class RenderInfo {
        protected final int density;
        protected final Vec4 referenceCenter;
        protected final float[] vertices;
        protected Position minElevation;
        protected Position maxElevation;

        protected RenderInfo(int n, float[] fArray, Vec4 vec4, Position position, Position position2) {
            this.density = n;
            this.referenceCenter = vec4;
            this.vertices = fArray;
            this.minElevation = position;
            this.maxElevation = position2;
        }

        protected long getSizeInBytes() {
            return 12 + this.vertices.length * 3 * 4;
        }
    }
}

