/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.worldwind.layers.rpf;

import gov.nasa.worldwind.avlist.AVList;
import gov.nasa.worldwind.avlist.AVListImpl;
import gov.nasa.worldwind.formats.rpf.RPFDataSeries;
import gov.nasa.worldwind.formats.rpf.RPFFrameFilename;
import gov.nasa.worldwind.formats.rpf.RPFFrameTransform;
import gov.nasa.worldwind.formats.rpf.RPFImageFile;
import gov.nasa.worldwind.formats.wvt.WaveletCodec;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.layers.rpf.RPFFileIndex;
import gov.nasa.worldwind.util.AbsentResourceList;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.WWIO;
import gov.nasa.worldwind.util.WWMath;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.logging.Level;

class RPFGenerator {
    private final RPFFileIndex fileIndex;
    private final FrameFile[] frameFiles;
    private final Sector globalBounds;
    private final AbsentResourceList absentFrames;
    private final int smallImageSize;
    private final int preloadRes;
    public static final String RPF_FILE_INDEX = "RPFGenerator.RPFFileIndex";
    public static final String WAVELET_IMAGE_THRESHOLD = "RPFGenerator.WaveletImageThreshold";
    public static final String WAVELET_PRELOAD_SIZE = "RPFGenerator.WaveletPreloadSize";

    public RPFGenerator(AVList aVList) {
        if (aVList == null) {
            String string = Logging.getMessage("nullValue.AVListIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        aVList = RPFGenerator.initParams(aVList.copy());
        this.fileIndex = (RPFFileIndex)aVList.getValue(RPF_FILE_INDEX);
        this.frameFiles = this.loadFrameFiles(this.fileIndex);
        this.globalBounds = this.computeGlobalBounds(this.fileIndex);
        this.absentFrames = new AbsentResourceList(1, 0);
        this.smallImageSize = (Integer)aVList.getValue(WAVELET_IMAGE_THRESHOLD);
        this.preloadRes = (Integer)aVList.getValue(WAVELET_PRELOAD_SIZE);
    }

    private static AVList initParams(AVList aVList) {
        if (aVList == null) {
            String string = Logging.getMessage("nullValue.AVListIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (aVList.getValue(RPF_FILE_INDEX) == null) {
            String string = "RPFFileIndex is null";
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        Object object = aVList.getValue(WAVELET_IMAGE_THRESHOLD);
        if (object == null || !(object instanceof Integer)) {
            aVList.setValue(WAVELET_IMAGE_THRESHOLD, 256);
        }
        if ((object = aVList.getValue(WAVELET_PRELOAD_SIZE)) == null || !(object instanceof Integer) || !WWMath.isPowerOfTwo((Integer)object)) {
            aVList.setValue(WAVELET_PRELOAD_SIZE, 32);
        }
        return aVList;
    }

    public Sector getGlobalBounds() {
        return this.globalBounds;
    }

    public RPFServiceInstance getServiceInstance() {
        return new RPFServiceInstance();
    }

    private FrameFile[] loadFrameFiles(RPFFileIndex rPFFileIndex) {
        ArrayList<FrameFile> arrayList = new ArrayList<FrameFile>();
        long l = -1L;
        RPFFileIndex.Table table = rPFFileIndex.getRPFFileTable();
        if (table != null) {
            for (RPFFileIndex.Record record : table.getRecords()) {
                Object object;
                ++l;
                RPFFileIndex.RPFFileRecord rPFFileRecord = (RPFFileIndex.RPFFileRecord)record;
                long l2 = record.getKey();
                long l3 = rPFFileRecord.getWaveletSecondaryKey();
                Sector sector = rPFFileRecord.getSector();
                if (l2 != -1L && l3 != -1L && sector != null) {
                    object = rPFFileIndex.getRPFFile(l2);
                    File file = rPFFileIndex.getWaveletFile(l3);
                    arrayList.add(new FrameFile(l, (File)object, file, sector));
                    continue;
                }
                object = "Ignoring frame file: " + (l2 != -1L ? rPFFileIndex.getRPFFile(l2).getPath() : "?");
                Logging.logger().fine((String)object);
            }
        }
        FrameFile[] frameFileArray = new FrameFile[arrayList.size()];
        arrayList.toArray(frameFileArray);
        return frameFileArray;
    }

    private Sector computeGlobalBounds(RPFFileIndex rPFFileIndex) {
        Sector sector = null;
        if (rPFFileIndex != null && rPFFileIndex.getIndexProperties() != null) {
            sector = rPFFileIndex.getIndexProperties().getBoundingSector();
        }
        if (sector == null) {
            sector = Sector.EMPTY_SECTOR;
        }
        return sector;
    }

    private void markFrameFileAbsent(FrameFile frameFile) {
        this.absentFrames.markResourceAbsent(frameFile.id);
    }

    private boolean isFrameFileAbsent(FrameFile frameFile) {
        return this.absentFrames.isResourceAbsent(frameFile.id);
    }

    private static class FrameFile {
        public long id;
        public File rpfFile;
        public File waveletFile;
        public Sector sector;
        public WaveletCodec codec;
        public RPFFrameFilename frameFile;
        public RPFFrameTransform transform;

        public FrameFile(long l, File file, File file2, Sector sector) {
            this.id = l;
            this.rpfFile = file;
            this.waveletFile = file2;
            this.sector = sector;
            this.frameFile = RPFFrameFilename.parseFilename(file.getName().toUpperCase());
        }

        public RPFFrameTransform getFrameTransform() {
            if (this.transform == null) {
                RPFDataSeries rPFDataSeries = RPFDataSeries.dataSeriesFor(this.frameFile.getDataSeriesCode());
                this.transform = RPFFrameTransform.createFrameTransform(this.frameFile.getZoneCode(), rPFDataSeries.rpfDataType, rPFDataSeries.scaleOrGSD);
            }
            return this.transform;
        }

        public int getFrameNumber() {
            return this.frameFile.getFrameNumber();
        }
    }

    public class RPFServiceInstance {
        public static final String BBOX = "bbox";
        public static final String WIDTH = "width";
        public static final String HEIGHT = "height";

        public BufferedImage serviceRequest(AVList aVList) throws IOException {
            if (aVList == null) {
                String string = Logging.getMessage("nullValue.AVListIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            String string = this.validate(aVList);
            if (string != null) {
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            try {
                Sector sector = (Sector)aVList.getValue(BBOX);
                int n = (Integer)aVList.getValue(WIDTH);
                int n2 = (Integer)aVList.getValue(HEIGHT);
                BufferedImage bufferedImage = new BufferedImage(n, n2, 6);
                int n3 = 0;
                for (FrameFile frameFile : RPFGenerator.this.frameFiles) {
                    block12: {
                        try {
                            if (!sector.intersects(frameFile.sector)) {
                            }
                            break block12;
                        }
                        catch (Exception exception) {
                            String string2 = String.format("Exception while computing frame bounds: %s", frameFile.rpfFile);
                            Logging.logger().log(Level.SEVERE, string2, exception);
                            RPFGenerator.this.markFrameFileAbsent(frameFile);
                        }
                        continue;
                    }
                    if (RPFGenerator.this.isFrameFileAbsent(frameFile)) continue;
                    Sector sector2 = frameFile.sector;
                    int n4 = (int)(sector2.getDeltaLonDegrees() * (double)bufferedImage.getWidth() / sector.getDeltaLonDegrees());
                    int n5 = (int)(sector2.getDeltaLatDegrees() * (double)bufferedImage.getHeight() / sector.getDeltaLatDegrees());
                    if (n4 > RPFGenerator.this.smallImageSize || n5 > RPFGenerator.this.smallImageSize) {
                        RPFFrameTransform.RPFImage[] rPFImageArray = this.getImageFromRPFSource(frameFile);
                        if (rPFImageArray == null) continue;
                        for (RPFFrameTransform.RPFImage rPFImage : rPFImageArray) {
                            if (rPFImage.getSector() == null || rPFImage.getImage() == null) continue;
                            this.drawImageIntoRequest(bufferedImage, sector, rPFImage.getImage(), rPFImage.getSector());
                        }
                    } else {
                        int n6 = n4;
                        n6 = n5 > n6 ? n5 : n6;
                        int n7 = (int)Math.ceil(Math.log(n6) / Math.log(2.0));
                        int n8 = (int)Math.pow(2.0, n7);
                        BufferedImage bufferedImage2 = this.getImageFromWaveletEncoding(frameFile, n8 = Math.max(1, n8));
                        if (bufferedImage2 == null) continue;
                        this.drawImageIntoRequest(bufferedImage, sector, bufferedImage2, sector2);
                    }
                    ++n3;
                }
                if (n3 <= 0) {
                    return null;
                }
                return bufferedImage;
            }
            catch (Exception exception) {
                String string3 = "Exception while processing request";
                Logging.logger().log(Level.SEVERE, string3, exception);
                throw new IOException(string3);
            }
        }

        private void drawImageIntoRequest(BufferedImage bufferedImage, Sector sector, BufferedImage bufferedImage2, Sector sector2) {
            double d = (sector2.getMinLongitude().degrees - sector.getMinLongitude().degrees) * ((double)bufferedImage.getWidth() / sector.getDeltaLonDegrees());
            double d2 = (sector.getMaxLatitude().degrees - sector2.getMaxLatitude().degrees) * ((double)bufferedImage.getHeight() / sector.getDeltaLatDegrees());
            double d3 = (double)bufferedImage.getWidth() / sector.getDeltaLonDegrees() * (sector2.getDeltaLonDegrees() / (double)bufferedImage2.getWidth());
            double d4 = (double)bufferedImage.getHeight() / sector.getDeltaLatDegrees() * (sector2.getDeltaLatDegrees() / (double)bufferedImage2.getHeight());
            Graphics2D graphics2D = (Graphics2D)bufferedImage.getGraphics();
            AffineTransform affineTransform = graphics2D.getTransform();
            graphics2D.translate(d, d2);
            graphics2D.scale(d3, d4);
            graphics2D.drawRenderedImage(bufferedImage2, null);
            graphics2D.setTransform(affineTransform);
        }

        public BufferedImage serviceRequest(URL uRL) throws IOException {
            if (uRL == null) {
                String string = Logging.getMessage("nullValue.URLIsNull");
                Logging.logger().severe(string);
                throw new IllegalArgumentException(string);
            }
            AVListImpl aVListImpl = new AVListImpl();
            String string = uRL.getQuery();
            if (string != null) {
                String[] stringArray;
                for (String string2 : stringArray = string.split("&")) {
                    String[] stringArray2 = string2.split("=", 2);
                    if (stringArray2 == null || stringArray2.length != 2) continue;
                    aVListImpl.setValue(stringArray2[0], stringArray2[1]);
                }
            }
            this.initParams(aVListImpl);
            return this.serviceRequest(aVListImpl);
        }

        private String validate(AVList aVList) {
            StringBuffer stringBuffer = new StringBuffer();
            Object object = aVList.getValue(BBOX);
            if (object == null || !(object instanceof Sector)) {
                stringBuffer.append("bounding box");
            }
            if ((object = aVList.getValue(WIDTH)) == null || !(object instanceof Integer) || (Integer)object < 1) {
                stringBuffer.append(WIDTH);
            }
            if ((object = aVList.getValue(HEIGHT)) == null || !(object instanceof Integer) || (Integer)object < 1) {
                stringBuffer.append(HEIGHT);
            }
            if (stringBuffer.length() == 0) {
                return null;
            }
            return "Inavlid RPFGenerator service request fields: " + stringBuffer.toString();
        }

        private AVList initParams(AVList aVList) {
            String[] stringArray;
            String string = aVList.getStringValue(BBOX);
            if (string != null && (stringArray = string.split(",")) != null && stringArray.length == 4) {
                try {
                    double d = Double.parseDouble(stringArray[0]);
                    double d2 = Double.parseDouble(stringArray[1]);
                    double d3 = Double.parseDouble(stringArray[2]);
                    double d4 = Double.parseDouble(stringArray[3]);
                    aVList.setValue(BBOX, Sector.fromDegrees(d2, d4, d, d3));
                }
                catch (NumberFormatException numberFormatException) {
                    Logging.logger().log(Level.WARNING, "Parameter conversion error", numberFormatException);
                    aVList.setValue(BBOX, null);
                }
            }
            if ((string = aVList.getStringValue(WIDTH)) != null) {
                try {
                    int n = Integer.parseInt(string);
                    aVList.setValue(WIDTH, n);
                }
                catch (NumberFormatException numberFormatException) {
                    Logging.logger().log(Level.WARNING, "Parameter conversion error", numberFormatException);
                    aVList.setValue(WIDTH, null);
                }
            }
            if ((string = aVList.getStringValue(HEIGHT)) != null) {
                try {
                    int n = Integer.parseInt(string);
                    aVList.setValue(HEIGHT, n);
                }
                catch (NumberFormatException numberFormatException) {
                    Logging.logger().log(Level.WARNING, "Parameter conversion error", numberFormatException);
                    aVList.setValue(HEIGHT, null);
                }
            }
            return aVList;
        }

        private RPFFrameTransform.RPFImage[] getImageFromRPFSource(FrameFile frameFile) {
            try {
                File file = frameFile.rpfFile;
                RPFImageFile rPFImageFile = RPFImageFile.load(file);
                BufferedImage bufferedImage = rPFImageFile.getBufferedImage();
                return frameFile.getFrameTransform().deproject(frameFile.getFrameNumber(), bufferedImage);
            }
            catch (Exception exception) {
                String string = "Exception while reading frame file: " + frameFile.rpfFile;
                Logging.logger().log(Level.SEVERE, string, exception);
                RPFGenerator.this.markFrameFileAbsent(frameFile);
                return null;
            }
        }

        private BufferedImage getImageFromWaveletEncoding(FrameFile frameFile, int n) {
            if (n <= 0) {
                return null;
            }
            try {
                WaveletCodec waveletCodec;
                Object object;
                if (n <= RPFGenerator.this.preloadRes) {
                    if (frameFile.codec == null) {
                        object = WWIO.readFileToBuffer(frameFile.waveletFile);
                        frameFile.codec = WaveletCodec.loadPartial((ByteBuffer)object, RPFGenerator.this.preloadRes);
                    }
                    waveletCodec = frameFile.codec;
                } else {
                    object = WWIO.readFileToBuffer(frameFile.waveletFile);
                    waveletCodec = WaveletCodec.loadPartial((ByteBuffer)object, n);
                }
                object = null;
                if (waveletCodec != null) {
                    object = waveletCodec.reconstruct(n);
                }
                return object;
            }
            catch (Exception exception) {
                String string = "Exception while reading wavelet file: " + frameFile.waveletFile;
                Logging.logger().log(Level.SEVERE, string, exception);
                RPFGenerator.this.markFrameFileAbsent(frameFile);
                return null;
            }
        }
    }
}

