/*
 * Decompiled with CFR 0.152.
 */
package de.gsi.math;

import de.gsi.dataset.AxisDescription;
import de.gsi.dataset.DataSet;
import de.gsi.dataset.DataSetError;
import de.gsi.dataset.DefaultNumberFormatter;
import de.gsi.dataset.EditableDataSet;
import de.gsi.dataset.Formatter;
import de.gsi.dataset.GridDataSet;
import de.gsi.dataset.Histogram;
import de.gsi.dataset.spi.DoubleDataSet;
import de.gsi.dataset.spi.DoubleErrorDataSet;
import de.gsi.dataset.spi.Histogram;
import de.gsi.dataset.spi.utils.DoublePointError;
import de.gsi.dataset.utils.NoDuplicatesList;
import de.gsi.math.ArrayMath;
import de.gsi.math.Math;
import de.gsi.math.MathBase;
import de.gsi.math.SimpleDataSetEstimators;
import de.gsi.math.TRandom;
import de.gsi.math.spectra.Apodization;
import de.gsi.math.spectra.SpectrumTools;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.jetbrains.annotations.NotNull;
import org.jtransforms.fft.DoubleFFT_1D;

public final class DataSetMath {
    private static final char INTEGRAL_SYMBOL = '\u222b';
    private static final char DIFFERENTIAL_SYMBOL = '\u2202';
    private static final char MULTIPLICATION_SYMBOL = '\u00b7';
    private static final String DIFFERENTIAL = "\u2202/\u2202x";
    public static Formatter<Number> DEFAULT_FORMATTER = new DefaultNumberFormatter();

    private DataSetMath() {
    }

    public static DoubleErrorDataSet applyMathOperation(DoubleErrorDataSet ret, MathOp op, double x1, double y1, double y2, double eyn1, double eyp1, double eyn2, double eyp2) {
        switch (op) {
            case ADD: {
                return ret.add(x1, y1 + y2, MathBase.hypot(eyn1, eyn2), MathBase.hypot(eyp1, eyp2));
            }
            case SUBTRACT: {
                return ret.add(x1, y1 - y2, MathBase.hypot(eyn1, eyn2), MathBase.hypot(eyp1, eyp2));
            }
            case MULTIPLY: {
                return ret.add(x1, y1 * y2, MathBase.hypot(y2 * eyn1, y1 * eyn2), MathBase.hypot(y2 * eyp1, y1 * eyp2));
            }
            case DIVIDE: {
                double newY = y1 / y2;
                double nEYN = MathBase.hypot(eyn1 / y2, newY * eyn2 / y2);
                double nEYP = MathBase.hypot(eyp1 / y2, newY * eyp2 / y2);
                return ret.add(x1, newY, nEYN, nEYP);
            }
            case SQR: {
                return ret.add(x1, MathBase.sqr(y1 + y2), 2.0 * MathBase.abs(y1 + y2) * MathBase.hypot(eyn1, eyn2), 2.0 * MathBase.abs(y1 + y2) * MathBase.hypot(eyp1, eyp2));
            }
            case SQRT: {
                return ret.add(x1, MathBase.sqrt(y1 + y2), MathBase.sqrt(MathBase.abs(y1 + y2)) * MathBase.hypot(eyn1, eyn2), MathBase.sqrt(MathBase.abs(y1 + y2)) * MathBase.hypot(eyp1, eyp2));
            }
            case LOG10: {
                double norm = 1.0 / MathBase.log(10.0);
                double nEYNLog = y1 + y2 > 0.0 ? norm / MathBase.abs(y1 + y2) * MathBase.hypot(eyn1, eyn2) : Double.NaN;
                double nEYPLog = y1 + y2 > 0.0 ? norm / MathBase.abs(y1 + y2) * MathBase.hypot(eyp1, eyp2) : Double.NaN;
                return ret.add(x1, 10.0 * MathBase.log10(y1 + y2), nEYNLog, nEYPLog);
            }
            case DB: {
                double normDb = 20.0 / MathBase.log(10.0);
                double nEYNDb = y1 + y2 > 0.0 ? normDb / MathBase.abs(y1 + y2) * MathBase.hypot(eyn1, eyn2) : Double.NaN;
                double nEYPDb = y1 + y2 > 0.0 ? normDb / MathBase.abs(y1 + y2) * MathBase.hypot(eyp1, eyp2) : Double.NaN;
                return ret.add(x1, 20.0 * MathBase.log10(y1 + y2), nEYNDb, nEYPDb);
            }
        }
        return ret.add(x1, y1 + y2, eyn1, eyp1);
    }

    private static double error(DataSet dataSet, ErrType eType, int index, double x, boolean interpolate) {
        if (!(dataSet instanceof DataSetError)) {
            return 0.0;
        }
        DataSetError ds = (DataSetError)dataSet;
        if (interpolate) {
            switch (eType) {
                case EXN: {
                    return ds.getErrorNegative(0, x);
                }
                case EXP: {
                    return ds.getErrorPositive(0, x);
                }
                case EYN: {
                    return ds.getErrorNegative(1, x);
                }
                case EYP: {
                    return ds.getErrorPositive(1, x);
                }
            }
        } else {
            switch (eType) {
                case EXN: {
                    return ds.getErrorNegative(0, index);
                }
                case EXP: {
                    return ds.getErrorPositive(0, index);
                }
                case EYN: {
                    return ds.getErrorNegative(1, index);
                }
                case EYP: {
                    return ds.getErrorPositive(1, index);
                }
            }
        }
        return 0.0;
    }

    @SafeVarargs
    public static DataSet addFunction(DataSet function1, DataSet function2, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function1, function2, MathOp.ADD, format);
    }

    @SafeVarargs
    public static DataSet addFunction(DataSet function, double value, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, value, MathOp.ADD, format);
    }

    @SafeVarargs
    public static DataSet addGaussianNoise(DataSet function, double sigma, Formatter<Number> ... format) {
        int nLength = function.getDataCount();
        String dataSetName = DataSetMath.getFormatter(format).format("{0}+noise({1})", new Object[]{function.getName(), sigma});
        DoubleErrorDataSet ret = new DoubleErrorDataSet(dataSetName, nLength);
        for (int i = 0; i < nLength; ++i) {
            double x = function.get(0, i);
            double y = function.get(1, i) + TRandom.Gaus(0.0, sigma);
            ret.add(x, y, sigma, sigma);
        }
        return ret;
    }

    @SafeVarargs
    public static DataSet averageDataSetsFIR(@NotNull List<DataSet> dataSets, int nUpdates, Formatter<Number> ... format) {
        if (dataSets.isEmpty()) {
            return new DoubleErrorDataSet(DataSetMath.getFormatter(format).format("LP({0}, FIR)", new Object[]{"<empty>"}));
        }
        String functionName = DataSetMath.getFormatter(format).format("LP({0}, FIR)", new Object[]{dataSets.get(0).getName()});
        if (dataSets.size() <= 1) {
            DataSet newFunction = dataSets.get(0);
            if (newFunction instanceof DataSetError) {
                return new DoubleErrorDataSet(functionName, newFunction.getValues(0), newFunction.getValues(1), DataSetMath.errors(newFunction, ErrType.EYN), DataSetMath.errors(newFunction, ErrType.EYP), newFunction.getDataCount(), true);
            }
            int ncount = newFunction.getDataCount();
            return new DoubleErrorDataSet(functionName, newFunction.getValues(0), newFunction.getValues(1), new double[ncount], new double[ncount], ncount, true);
        }
        int nAvg = MathBase.min(nUpdates, dataSets.size());
        DataSet newFunction = dataSets.get(dataSets.size() - 1);
        DoubleErrorDataSet retFunction = new DoubleErrorDataSet(functionName, newFunction.getDataCount() + 2);
        for (int i = 0; i < newFunction.getDataCount(); ++i) {
            double newX = newFunction.get(0, i);
            double mean = 0.0;
            double variance = 0.0;
            double eyn = 0.0;
            double eyp = 0.0;
            int count = 0;
            for (int j = MathBase.max(0, dataSets.size() - nAvg); j < dataSets.size(); ++j) {
                DataSet oldFunction = dataSets.get(j);
                double oldX = oldFunction.get(0, i);
                double oldY = oldX == newX ? oldFunction.get(1, i) : oldFunction.getValue(0, new double[]{newX});
                mean += oldY;
                variance += oldY * oldY;
                boolean inter = oldX != newX;
                eyn += DataSetMath.error(oldFunction, ErrType.EYN, i, newX, inter);
                eyp += DataSetMath.error(oldFunction, ErrType.EYP, i, newX, inter);
                ++count;
            }
            if (count == 0) {
                retFunction.add(newX, Double.NaN, Double.NaN, Double.NaN);
                continue;
            }
            eyn /= (double)count;
            eyp /= (double)count;
            double mean2 = (mean /= (double)count) * mean;
            double diff = MathBase.abs((variance /= (double)count) - mean2);
            eyn = MathBase.sqrt(eyn * eyn + diff);
            eyp = MathBase.sqrt(eyp * eyp + diff);
            retFunction.add(newX, mean, eyn, eyp);
        }
        return retFunction;
    }

    @SafeVarargs
    public static DataSet averageDataSetsIIR(DataSet prevAverage, DataSet prevAverage2, DataSet newDataSet, int nUpdates, Formatter<Number> ... format) {
        String functionName = DataSetMath.getFormatter(format).format("LP({0}, IIR)", new Object[]{newDataSet.getName()});
        if (prevAverage == null || prevAverage2 == null || prevAverage.getDataCount() == 0 || prevAverage2.getDataCount() == 0) {
            double[] yValues = newDataSet.getValues(1);
            double[] eyn = DataSetMath.errors(newDataSet, ErrType.EYN);
            double[] eyp = DataSetMath.errors(newDataSet, ErrType.EYP);
            if (prevAverage2 instanceof DoubleErrorDataSet) {
                ((DoubleErrorDataSet)prevAverage2).set(newDataSet.getValues(0), ArrayMath.sqr(yValues), ArrayMath.sqr(eyn), ArrayMath.sqr(eyp));
            } else if (prevAverage2 instanceof DoubleDataSet) {
                ((DoubleDataSet)prevAverage2).set(newDataSet.getValues(0), ArrayMath.sqr(yValues));
            }
            return new DoubleErrorDataSet(functionName, newDataSet.getValues(0), yValues, eyn, eyp, newDataSet.getDataCount(), true);
        }
        int dataCount1 = prevAverage.getDataCount();
        int dataCount2 = prevAverage2.getDataCount();
        DoubleErrorDataSet retFunction = dataCount1 == 0 ? new DoubleErrorDataSet(functionName, newDataSet.getValues(0), newDataSet.getValues(1), DataSetMath.errors(newDataSet, ErrType.EYN), DataSetMath.errors(newDataSet, ErrType.EYP), newDataSet.getDataCount(), true) : new DoubleErrorDataSet(prevAverage.getName(), prevAverage.getValues(0), prevAverage.getValues(1), DataSetMath.errors(prevAverage, ErrType.EYN), DataSetMath.errors(prevAverage, ErrType.EYP), newDataSet.getDataCount(), true);
        double alpha = 1.0 / (1.0 + (double)nUpdates);
        boolean avg2Empty = dataCount2 == 0;
        for (int i = 0; i < dataCount1; ++i) {
            double oldX = prevAverage.get(0, i);
            double oldY = prevAverage.get(1, i);
            double oldY2 = avg2Empty ? oldY * oldY : prevAverage2.get(1, i);
            double newX = newDataSet.get(0, i);
            boolean inter = oldX != newX;
            double y = inter ? newDataSet.getValue(1, new double[]{oldX}) : newDataSet.get(1, i);
            double newVal = (1.0 - alpha) * oldY + alpha * y;
            double newVal2 = (1.0 - alpha) * oldY2 + alpha * (y * y);
            double eyn = DataSetMath.error(newDataSet, ErrType.EYN, i, newX, inter);
            double eyp = DataSetMath.error(newDataSet, ErrType.EYP, i, newX, inter);
            if (prevAverage2 instanceof DoubleErrorDataSet) {
                if (avg2Empty) {
                    ((DoubleErrorDataSet)prevAverage2).add(newX, newVal2, eyn, eyp);
                } else {
                    ((DoubleErrorDataSet)prevAverage2).set(i, newX, newVal2, eyn, eyp);
                }
            }
            double newEYN = MathBase.sqrt(MathBase.abs(newVal2 - MathBase.pow(newVal, 2.0)) + eyn * eyn);
            double newEYP = MathBase.sqrt(MathBase.abs(newVal2 - MathBase.pow(newVal, 2.0)) + eyp * eyp);
            retFunction.set(i, oldX, newVal, newEYN, newEYP);
        }
        return retFunction;
    }

    @SafeVarargs
    public static DataSet dbFunction(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, 0.0, MathOp.DB, format);
    }

    @SafeVarargs
    public static DataSet dbFunction(DataSet function1, DataSet function2, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function1, function2, MathOp.DB, format);
    }

    @SafeVarargs
    public static DataSet derivativeFunction(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.derivativeFunction(function, 1.0, format);
    }

    @SafeVarargs
    public static DataSet derivativeFunction(DataSet function, double sign, Formatter<Number> ... format) {
        double x0;
        int i;
        String signAdd = sign == 1.0 ? "" : Double.toString(sign) + "\u00b7";
        int ncount = function.getDataCount();
        String functionName = DataSetMath.getFormatter(format).format("{0}{1}({2})", new Object[]{signAdd, DIFFERENTIAL, function.getName()});
        DoubleErrorDataSet retFunction = new DoubleErrorDataSet(functionName, ncount);
        if (ncount <= 3) {
            return retFunction;
        }
        for (i = 0; i < 2; ++i) {
            x0 = function.get(0, i);
            retFunction.add(x0, 0.0, 0.0, 0.0);
        }
        for (i = 2; i < ncount - 2; ++i) {
            x0 = function.get(0, i);
            double stepL = x0 - function.get(0, i - 1);
            double stepR = function.get(0, i + 1) - x0;
            double valL = function.get(1, i - 1);
            double valC = function.get(1, i);
            double valR = function.get(1, i + 1);
            double yenL = DataSetMath.error(function, ErrType.EYN, i - 1);
            double yenC = DataSetMath.error(function, ErrType.EYN, i);
            double yenR = DataSetMath.error(function, ErrType.EYN, i + 1);
            double yen = MathBase.sqrt(MathBase.sqr(yenL) + MathBase.sqr(yenC) + MathBase.sqr(yenR)) / 4.0;
            double yepL = DataSetMath.error(function, ErrType.EYP, i - 1);
            double yepC = DataSetMath.error(function, ErrType.EYP, i);
            double yepR = DataSetMath.error(function, ErrType.EYP, i + 1);
            double yep = MathBase.sqrt(MathBase.sqr(yepL) + MathBase.sqr(yepC) + MathBase.sqr(yepR)) / 4.0;
            double derivative = 0.5 * ((valC - valL) / stepL + (valR - valC) / stepR);
            retFunction.add(x0, sign * derivative, yen, yep);
        }
        for (i = ncount - 2; i < ncount; ++i) {
            x0 = function.get(0, i);
            retFunction.add(x0, 0.0, 0.0, 0.0);
        }
        return retFunction;
    }

    @SafeVarargs
    public static DataSet divideFunction(DataSet function1, DataSet function2, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function1, function2, MathOp.DIVIDE, format);
    }

    @SafeVarargs
    public static DataSet divideFunction(DataSet function, double value, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, value, MathOp.DIVIDE, format);
    }

    public static double error(DataSet dataSet, ErrType eType, double x) {
        return DataSetMath.error(dataSet, eType, -1, x, true);
    }

    public static double error(DataSet dataSet, ErrType eType, int index) {
        return DataSetMath.error(dataSet, eType, index, 0.0, false);
    }

    public static double[] errors(DataSet dataSet, ErrType eType) {
        int nDim = dataSet.getDataCount();
        if (!(dataSet instanceof DataSetError)) {
            return new double[nDim];
        }
        DataSetError ds = (DataSetError)dataSet;
        switch (eType) {
            case EXN: {
                return DataSetMath.cropToLength(ds.getErrorsNegative(0), nDim);
            }
            case EXP: {
                return DataSetMath.cropToLength(ds.getErrorsPositive(0), nDim);
            }
            case EYN: {
                return DataSetMath.cropToLength(ds.getErrorsNegative(1), nDim);
            }
        }
        return DataSetMath.cropToLength(ds.getErrorsPositive(1), nDim);
    }

    @SafeVarargs
    public static DataSet filterFunction(DataSet function, double width, Filter filterType, Formatter<Number> ... format) {
        int n = function.getDataCount();
        String dataSetName = DataSetMath.getFormatter(format).format("{0}({1},{2})", new Object[]{filterType.getTag(), function.getName(), width});
        DoubleErrorDataSet filteredFunction = new DoubleErrorDataSet(dataSetName, n);
        for (int dim = 0; dim < filteredFunction.getDimension(); ++dim) {
            AxisDescription refAxisDescription = function.getAxisDescription(dim);
            filteredFunction.getAxisDescription(dim).set(refAxisDescription.getName(), new String[]{refAxisDescription.getUnit()});
        }
        double[] subArrayY = new double[n];
        double[] subArrayYn = new double[n];
        double[] subArrayYp = new double[n];
        double[] xValues = function.getValues(0);
        double[] yValues = function.getValues(1);
        double[] yen = DataSetMath.errors(function, ErrType.EYN);
        double[] yep = DataSetMath.errors(function, ErrType.EYN);
        block9: for (int i = 0; i < n; ++i) {
            double time0 = xValues[i];
            int count = 0;
            for (int j = 0; j < n; ++j) {
                double time = xValues[j];
                if (!(MathBase.abs(time0 - time) <= width)) continue;
                subArrayY[count] = yValues[j];
                subArrayYn[count] = yen[j];
                subArrayYp[count] = yep[j];
                ++count;
            }
            double norm = count > 0 ? 1.0 / MathBase.sqrt(count) : 0.0;
            switch (filterType) {
                case MEDIAN: {
                    filteredFunction.add(time0, Math.median(subArrayY, count), Math.median(subArrayYn, count), Math.median(subArrayYp, count));
                    continue block9;
                }
                case MIN: {
                    filteredFunction.add(time0, Math.minimum(subArrayY, count), Math.minimum(subArrayYn, count), Math.minimum(subArrayYp, count));
                    continue block9;
                }
                case MAX: {
                    filteredFunction.add(time0, Math.maximum(subArrayY, count), Math.maximum(subArrayYn, count), Math.maximum(subArrayYp, count));
                    continue block9;
                }
                case P2P: {
                    filteredFunction.add(time0, Math.peakToPeak(subArrayY, count), Math.peakToPeak(subArrayYn, count), Math.peakToPeak(subArrayYp, count));
                    continue block9;
                }
                case RMS: {
                    filteredFunction.add(time0, Math.rms(subArrayY, count), Math.rms(subArrayYn, count), Math.rms(subArrayYp, count));
                    continue block9;
                }
                case GEOMMEAN: {
                    filteredFunction.add(time0, Math.geometricMean(subArrayY, 0, count), Math.geometricMean(subArrayYn, 0, count), Math.geometricMean(subArrayYp, 0, count));
                    continue block9;
                }
                default: {
                    filteredFunction.add(time0, Math.mean(subArrayY, count), Math.mean(subArrayYn, count) * norm, Math.mean(subArrayYp, count) * norm);
                }
            }
        }
        return filteredFunction;
    }

    @SafeVarargs
    public static DataSet geometricMeanFilteredFunction(DataSet function, double width, Formatter<Number> ... format) {
        return DataSetMath.filterFunction(function, width, Filter.GEOMMEAN, format);
    }

    @SafeVarargs
    public static DataSet getSubRange(DataSet function, double xMin, double xMax, Formatter<Number> ... format) {
        int nLength = function.getDataCount();
        String dataSetName = DataSetMath.getFormatter(format).format("subRange({0}, {1})", new Object[]{xMin, xMax});
        DoubleErrorDataSet ret = new DoubleErrorDataSet(dataSetName, nLength);
        for (int i = 0; i < nLength; ++i) {
            double x = function.get(0, i);
            double y = function.get(1, i);
            double ex = DataSetMath.error(function, ErrType.EXP, i);
            double ey = DataSetMath.error(function, ErrType.EYP, i);
            if (!(x >= xMin) || !(x <= xMax)) continue;
            ret.add(x, y, ex, ey);
        }
        return ret;
    }

    @SafeVarargs
    public static DataSet iirLowPassFilterFunction(DataSet function, double width, Formatter<Number> ... format) {
        double y;
        double x1;
        double x0;
        int i;
        int n = function.getDataCount();
        String dataSetName = DataSetMath.getFormatter(format).format("iir{0}({1},{2})", new Object[]{Filter.MEAN.getTag(), function.getName(), width});
        DoubleErrorDataSet filteredFunction = new DoubleErrorDataSet(dataSetName, n);
        if (n <= 1) {
            int index;
            if (!(function instanceof GridDataSet)) {
                filteredFunction.set(function);
                return filteredFunction;
            }
            filteredFunction.set(function.getValues(0), function.getValues(1), DataSetMath.errors(function, ErrType.EYN), DataSetMath.errors(function, ErrType.EYP));
            for (index = 0; index < function.getDataCount(); ++index) {
                String label = function.getDataLabel(index);
                if (label == null || label.isEmpty()) continue;
                filteredFunction.addDataLabel(index, label);
            }
            for (index = 0; index < function.getDataCount(); ++index) {
                String style = function.getStyle(index);
                if (style == null || style.isEmpty()) continue;
                filteredFunction.addDataStyle(index, style);
            }
            filteredFunction.setStyle(function.getStyle());
            return filteredFunction;
        }
        double[] xValues = function.getValues(0);
        double[] yValues = function.getValues(1);
        double[] yen = DataSetMath.errors(function, ErrType.EYN);
        double[] yep = DataSetMath.errors(function, ErrType.EYN);
        double[] yUp = new double[n];
        double[] yDown = new double[n];
        double[] ye1 = new double[n];
        double[] ye2 = new double[n];
        double smoothing = 0.5 * width;
        double smoothed = yValues[0];
        double smoothed2 = smoothed * smoothed;
        for (i = 1; i < n; ++i) {
            x0 = xValues[i - 1];
            x1 = xValues[i];
            y = yValues[i];
            smoothed += (x1 - x0) * (y - smoothed) / smoothing;
            smoothed2 += (x1 - x0) * (y * y - smoothed2) / smoothing;
            yUp[i] = smoothed;
            ye1[i] = smoothed2;
        }
        smoothed = yValues[n - 1];
        smoothed2 = smoothed * smoothed;
        for (i = n - 2; i >= 0; --i) {
            x0 = xValues[i];
            x1 = xValues[i + 1];
            y = yValues[i];
            smoothed += (x1 - x0) * (y - smoothed) / smoothing;
            smoothed2 += (x1 - x0) * (y * y - smoothed2) / smoothing;
            yDown[i] = smoothed;
            ye2[i] = smoothed2;
        }
        filteredFunction.add(xValues[0], yValues[0], yen[0], yep[0]);
        for (i = 1; i < n; ++i) {
            double x12 = xValues[i];
            double y2 = 0.5 * (yUp[i] + yDown[i]);
            double mean2 = y2 * y2;
            double y22 = 0.5 * MathBase.pow(ye1[i] + ye2[i], 1.0);
            double avgError2 = MathBase.abs(y22 - mean2);
            double newEYN = MathBase.sqrt(avgError2 + yen[i] * yen[i]);
            double newEYP = MathBase.sqrt(avgError2 + yep[i] * yep[i]);
            filteredFunction.add(x12, y2, newEYN, newEYP);
        }
        return filteredFunction;
    }

    public static DoublePointError integral(DataSet function) {
        DataSet integratedFunction = DataSetMath.integrateFunction(function, new Formatter[0]);
        int lastPoint = integratedFunction.getDataCount() - 1;
        if (lastPoint <= 0) {
            return new DoublePointError(0.0, 0.0, 0.0, 0.0);
        }
        double x = integratedFunction.get(0, lastPoint);
        double y = integratedFunction.get(1, lastPoint);
        double yen = DataSetMath.error(integratedFunction, ErrType.EYN, lastPoint);
        double yep = DataSetMath.error(integratedFunction, ErrType.EYP, lastPoint);
        double ye = 0.5 * (yen + yep);
        return new DoublePointError(x, y, 0.0, ye);
    }

    public static DoublePointError integral(DataSet function, double xMin, double xMax) {
        DataSet integratedFunction = DataSetMath.integrateFunction(function, xMin, xMax, new Formatter[0]);
        int lastPoint = integratedFunction.getDataCount() - 1;
        double yen = DataSetMath.error(integratedFunction, ErrType.EYN, lastPoint);
        double yep = DataSetMath.error(integratedFunction, ErrType.EYP, lastPoint);
        double ye = 0.5 * (yen + yep);
        return new DoublePointError(integratedFunction.get(0, lastPoint), integratedFunction.get(1, lastPoint), 0.0, ye);
    }

    public static double integralSimple(DataSet function) {
        double integral1 = 0.0;
        double integral2 = 0.0;
        int nCount = function.getDataCount();
        if (nCount <= 1) {
            return 0.0;
        }
        for (int i = 1; i < nCount; ++i) {
            double step = function.get(0, i) - function.get(0, i - 1);
            double val1 = function.get(1, i - 1);
            double val2 = function.get(1, i);
            integral1 += step * val1;
            integral2 += step * val2;
        }
        return 0.5 * (integral1 + integral2);
    }

    @SafeVarargs
    public static DataSet integrateFunction(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.integrateFunction(function, Double.NEGATIVE_INFINITY, Double.POSITIVE_INFINITY, format);
    }

    @SafeVarargs
    public static DataSet integrateFunction(DataSet function, double xMin, double xMax, Formatter<Number> ... format) {
        double ep1;
        double en1;
        double step;
        double val2;
        int nLength = function.getDataCount();
        String pattern = "{0}({1})dyn|_'{'{2}'}'^'{'{3}'}'";
        String dataSetName = DataSetMath.getFormatter(format).format("{0}({1})dyn|_'{'{2}'}'^'{'{3}'}'", new Object[]{Character.valueOf('\u222b'), function.getName(), xMin, xMax});
        if (nLength <= 0) {
            if (!(function instanceof GridDataSet) || function.getDimension() > 2) {
                int ncount = function.getDataCount();
                double[] emptyVector = new double[ncount];
                return new DoubleErrorDataSet(dataSetName, function.getValues(0), emptyVector, emptyVector, emptyVector, ncount, true);
            }
            throw new IllegalStateException("not yet implemented for non 2D dataSets");
        }
        if (!function.getAxisDescription(0).isDefined()) {
            function.recomputeLimits(0);
        }
        double xMinLocal = function.getAxisDescription(0).getMin();
        double xMaxLocal = function.getAxisDescription(0).getMax();
        double sign = 1.0;
        if (Double.isFinite(xMin) && Double.isFinite(xMax)) {
            xMinLocal = MathBase.min(xMin, xMax);
            xMaxLocal = MathBase.max(xMin, xMax);
            if (xMin > xMax) {
                sign = -1.0;
            }
        } else if (Double.isFinite(xMin)) {
            xMinLocal = xMin;
        } else if (Double.isFinite(xMax)) {
            xMaxLocal = xMax;
        }
        DoubleErrorDataSet retFunction = new DoubleErrorDataSet(dataSetName, nLength);
        if (nLength <= 1) {
            retFunction.add(function.get(0, 0), 0.0, 0.0, 0.0);
            return retFunction;
        }
        double integral = 0.0;
        double integralEN = 0.0;
        double integralEP = 0.0;
        if (Double.isFinite(xMin) && xMin <= function.get(0, 0)) {
            double val1 = function.getValue(1, new double[]{xMin});
            double x1 = function.get(0, 0);
            val2 = function.get(1, 0);
            step = x1 - xMin;
            en1 = DataSetMath.error(function, ErrType.EYN, 0);
            ep1 = DataSetMath.error(function, ErrType.EYP, 0);
            integralEN = MathBase.hypot(integralEN, step * en1);
            integralEP = MathBase.hypot(integralEP, step * ep1);
            retFunction.add(xMin, integral += sign * 0.5 * step * (val1 + val2), 0.0, 0.0);
        }
        retFunction.add(function.get(0, 0), integral, integralEN, integralEP);
        for (int i = 1; i < nLength; ++i) {
            double x0 = function.get(0, i - 1);
            double x1 = function.get(0, i);
            double step2 = x1 - x0;
            double y0 = function.get(1, i - 1);
            double en12 = DataSetMath.error(function, ErrType.EYN, i - 1);
            double ep12 = DataSetMath.error(function, ErrType.EYP, i - 1);
            double y1 = function.get(1, i);
            double en2 = DataSetMath.error(function, ErrType.EYN, i);
            double ep2 = DataSetMath.error(function, ErrType.EYP, i);
            if (x0 >= xMinLocal && x1 <= xMaxLocal) {
                integral += sign * 0.5 * step2 * (y0 + y1);
                integralEN = MathBase.hypot(integralEN, 0.5 * step2 * (en12 + en2));
                integralEP = MathBase.hypot(integralEP, 0.5 * step2 * (ep12 + ep2));
            } else if (!(x1 < xMinLocal) || !(x0 < xMinLocal)) {
                if (x0 < xMinLocal && x1 > xMinLocal) {
                    retFunction.add(xMin, integral, integralEN, integralEP);
                    step2 = x1 - xMinLocal;
                    integral += sign * 0.5 * step2 * (function.getValue(1, new double[]{xMinLocal}) + y1);
                    integralEN = MathBase.hypot(integralEN, 0.5 * step2 * (en12 + en2));
                    integralEP = MathBase.hypot(integralEP, 0.5 * step2 * (ep12 + ep2));
                } else if (x0 < xMaxLocal && x1 > xMaxLocal) {
                    step2 = xMaxLocal - x0;
                    double yAtMax = function.getValue(1, new double[]{xMaxLocal});
                    integralEN = MathBase.hypot(integralEN, 0.5 * step2 * (en12 + en2));
                    integralEP = MathBase.hypot(integralEP, 0.5 * step2 * (ep12 + ep2));
                    retFunction.add(xMaxLocal, integral += sign * 0.5 * step2 * (y0 + yAtMax), integralEN, integralEP);
                }
            }
            retFunction.add(x1, integral, integralEN, integralEP);
        }
        if (Double.isFinite(xMax) && xMax > function.get(0, nLength - 1)) {
            double x0 = function.get(0, nLength - 1);
            double val1 = function.get(1, nLength - 1);
            val2 = function.getValue(1, new double[]{xMax});
            step = xMax - x0;
            en1 = DataSetMath.error(function, ErrType.EYN, nLength - 1);
            ep1 = DataSetMath.error(function, ErrType.EYP, nLength - 1);
            integralEN = MathBase.hypot(integralEN, step * en1);
            integralEP = MathBase.hypot(integralEP, step * ep1);
            retFunction.add(xMax, integral += 0.5 * step * (val1 + val2), integralEN, integralEP);
        }
        return retFunction;
    }

    @SafeVarargs
    public static DataSet integrateFromCentre(@NotNull DataSet function, double centre, double width, boolean normalise, Formatter<Number> ... format) {
        double xMinLocal = function.getAxisDescription(0).getMin();
        double xMaxLocal = function.getAxisDescription(0).getMax();
        double centreLocal = Double.isFinite(centre) ? centre : SimpleDataSetEstimators.computeCentreOfMass(function, 0, function.getDataCount());
        String pattern = "{0}({1},c={2})dyn|_'{'c-{3}'}'^'{'c+{3}'}'";
        int nLength = function.getDataCount();
        String dataSetName = DataSetMath.getFormatter(format).format("{0}({1},c={2})dyn|_'{'c-{3}'}'^'{'c+{3}'}'", new Object[]{Character.valueOf('\u222b'), function.getName(), centreLocal, width});
        if (nLength < 2) {
            if (!(function instanceof GridDataSet) || function.getDimension() > 2) {
                int ncount = function.getDataCount();
                double[] emptyVector = new double[ncount];
                return new DoubleErrorDataSet(dataSetName, function.getValues(0), emptyVector, emptyVector, emptyVector, ncount, true);
            }
            throw new IllegalStateException("not yet implemented for non 2D dataSets");
        }
        DoubleErrorDataSet retFunction = new DoubleErrorDataSet(dataSetName, nLength / 2);
        if (width <= 0.0) {
            return retFunction;
        }
        if (centreLocal <= xMinLocal || centreLocal >= xMaxLocal) {
            throw new IllegalArgumentException(String.format("centre %f is outside DataSetRange [%f,%f]", centreLocal, xMinLocal, xMaxLocal));
        }
        double scaleLocal = normalise ? 1.0 / DataSetMath.integral(function, MathBase.max(centreLocal - width, xMinLocal), MathBase.min(centreLocal + width, xMaxLocal)).getY() : 1.0;
        int centreLocalIndex = function.getIndex(0, new double[]{centreLocal});
        int maxIntIndex = MathBase.min(centreLocalIndex, function.getDataCount() - 1 - centreLocalIndex);
        double integral = 0.0;
        for (int i = 0; i < maxIntIndex; ++i) {
            double xL0 = function.get(0, centreLocalIndex - i - 1);
            double xL1 = function.get(0, centreLocalIndex - i);
            double stepL = xL1 - xL0;
            double xR0 = function.get(0, centreLocalIndex + i);
            double xR1 = function.get(0, centreLocalIndex + i + 1);
            double stepR = xR1 - xR0;
            double yL0 = function.get(1, centreLocalIndex - i - 1);
            double yL1 = function.get(1, centreLocalIndex - i);
            double yR0 = function.get(1, centreLocalIndex + i);
            double yR1 = function.get(1, centreLocalIndex + i + 1);
            double distanceFromCentre = 0.5 * (xR1 - xL0);
            retFunction.add(distanceFromCentre, integral += scaleLocal * 0.5 * (stepL * (yL0 + yL1) + stepR * (yR0 + yR1)), 0.0, 0.0);
        }
        return retFunction;
    }

    public static double integralWidth(@NotNull DataSet function, double centre, double maxWidth, double threshold) {
        return SimpleDataSetEstimators.getZeroCrossing(DataSetMath.integrateFromCentre(function, centre, maxWidth, true, new Formatter[0]), threshold);
    }

    @SafeVarargs
    public static DataSet inversedbFunction(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, 1.0, MathOp.INV_DB, format);
    }

    @SafeVarargs
    public static DataSet log10Function(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, 0.0, MathOp.LOG10, format);
    }

    @SafeVarargs
    public static DataSet log10Function(DataSet function1, DataSet function2, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function1, function2, MathOp.LOG10, format);
    }

    @SafeVarargs
    public static DataSet lowPassFilterFunction(DataSet function, double width, Formatter<Number> ... format) {
        return DataSetMath.filterFunction(function, width, Filter.MEAN, format);
    }

    @SafeVarargs
    public static DataSet magnitudeSpectrum(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.magnitudeSpectrum(function, Apodization.Hann, false, false, format);
    }

    @SafeVarargs
    public static DataSet magnitudeSpectrum(DataSet function, Apodization apodization, boolean dbScale, boolean normalisedFrequency, Formatter<Number> ... format) {
        int n = function.getDataCount();
        DoubleFFT_1D fastFourierTrafo = new DoubleFFT_1D((long)n);
        double[] fftSpectra = new double[n];
        for (int i = 0; i < n; ++i) {
            double window = apodization.getIndex(i, n);
            fftSpectra[i] = function.get(1, i) * window;
        }
        fastFourierTrafo.realForward(fftSpectra);
        double[] mag = dbScale ? SpectrumTools.computeMagnitudeSpectrum_dB(fftSpectra, true) : SpectrumTools.computeMagnitudeSpectrum(fftSpectra, true);
        double dt = function.get(0, function.getDataCount() - 1) - function.get(0, 0);
        double fsampling = normalisedFrequency || dt <= 0.0 ? 0.5 / (double)mag.length : 1.0 / dt;
        String functionName = DataSetMath.getFormatter(format).format("Mag{0}({1})", new Object[]{dbScale ? "[dB]" : "", function.getName()});
        DoubleErrorDataSet ret = new DoubleErrorDataSet(functionName, mag.length);
        for (int i = 0; i < mag.length; ++i) {
            ret.add((double)i * fsampling, mag[i], 0.0, 0.0);
        }
        return ret;
    }

    @SafeVarargs
    public static DataSet magnitudeSpectrumComplex(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.magnitudeSpectrumComplex(function, Apodization.Hann, false, false, format);
    }

    @SafeVarargs
    public static DataSet magnitudeSpectrumComplex(DataSet function, Apodization apodization, boolean dbScale, boolean normalisedFrequency, Formatter<Number> ... format) {
        int n = function.getDataCount();
        DoubleFFT_1D fastFourierTrafo = new DoubleFFT_1D((long)n);
        double[] fftSpectra = new double[2 * n];
        for (int i = 0; i < n; ++i) {
            double window = apodization.getIndex(i, n);
            fftSpectra[2 * i] = function.get(1, i) * window;
            fftSpectra[2 * i + 1] = function.get(2, i) * window;
        }
        fastFourierTrafo.complexForward(fftSpectra);
        double[] mag = dbScale ? SpectrumTools.computeMagnitudeSpectrum_dB(fftSpectra, true) : SpectrumTools.computeMagnitudeSpectrum(fftSpectra, true);
        double dt = function.get(0, function.getDataCount() - 1) - function.get(0, 0);
        double fsampling = normalisedFrequency || dt <= 0.0 ? 0.5 / (double)mag.length : 1.0 / dt;
        String functionName = DataSetMath.getFormatter(format).format("Mag{0}({1})", new Object[]{dbScale ? "[dB]" : "", function.getName()});
        DoubleErrorDataSet ret = new DoubleErrorDataSet(functionName, mag.length);
        for (int i = 0; i < mag.length; ++i) {
            if (i < mag.length / 2) {
                ret.add(((double)i - (double)mag.length / 2.0) * fsampling, mag[i + mag.length / 2], 0.0, 0.0);
                continue;
            }
            ret.add(((double)i - (double)mag.length / 2.0) * fsampling, mag[i - mag.length / 2], 0.0, 0.0);
        }
        return ret;
    }

    public static DataSet magnitudeSpectrumDecibel(DataSet function) {
        return DataSetMath.magnitudeSpectrum(function, Apodization.Hann, true, false, new Formatter[0]);
    }

    public static boolean sameHorizontalBase(DataSet function1, DataSet function2) {
        if (function1.getDataCount() != function2.getDataCount()) {
            return false;
        }
        for (int i = 0; i < function1.getDataCount(); ++i) {
            double X2;
            double X1 = function1.get(0, i);
            if (X1 == (X2 = function2.get(0, i))) continue;
            return false;
        }
        return true;
    }

    @SafeVarargs
    public static DataSet mathFunction(DataSet function1, DataSet function2, MathOp op, Formatter<Number> ... format) {
        boolean needsInterpolation;
        String newDataSetName = DataSetMath.getFormatter(format).format("{0}{1}{2}", new Object[]{function1.getName(), op.getTag(), function2.getName()});
        DoubleErrorDataSet ret = new DoubleErrorDataSet(newDataSetName, function1.getDataCount());
        ret.getAxisDescription(0).set(function1.getAxisDescription(0));
        ret.getAxisDescription(1).set(function1.getAxisDescription(1).getName(), new String[]{function1.getAxisDescription(1).getUnit()});
        boolean bl = needsInterpolation = !DataSetMath.sameHorizontalBase(function1, function2);
        if (needsInterpolation) {
            List<Double> xValues = DataSetMath.getCommonBase(function1, function2);
            for (double x : xValues) {
                double Y1 = function1.getValue(1, new double[]{x});
                double Y2 = function2.getValue(1, new double[]{x});
                double eyn1 = DataSetMath.error(function1, ErrType.EYN, 0, x, needsInterpolation);
                double eyp1 = DataSetMath.error(function1, ErrType.EYP, 0, x, needsInterpolation);
                double eyn2 = DataSetMath.error(function2, ErrType.EYN, 0, x, needsInterpolation);
                double eyp2 = DataSetMath.error(function2, ErrType.EYP, 0, x, needsInterpolation);
                DataSetMath.applyMathOperation(ret, op, x, Y1, Y2, eyn1, eyp1, eyn2, eyp2);
            }
            return ret;
        }
        for (int i = 0; i < function1.getDataCount(); ++i) {
            double X1 = function1.get(0, i);
            double Y1 = function1.get(1, i);
            double Y2 = function2.get(1, i);
            double eyn1 = DataSetMath.error(function1, ErrType.EYN, i);
            double eyp1 = DataSetMath.error(function1, ErrType.EYP, i);
            double eyn2 = DataSetMath.error(function2, ErrType.EYN, i, X1, needsInterpolation);
            double eyp2 = DataSetMath.error(function2, ErrType.EYP, i, X1, needsInterpolation);
            DataSetMath.applyMathOperation(ret, op, X1, Y1, Y2, eyn1, eyp1, eyn2, eyp2);
        }
        return ret;
    }

    public static List<Double> getCommonBase(DataSet ... functions) {
        NoDuplicatesList xValues = new NoDuplicatesList();
        for (DataSet function : functions) {
            for (int i = 0; i < function.getDataCount(); ++i) {
                if (function instanceof Histogram) {
                    xValues.add(((Histogram)function).getBinLimits(0, Histogram.Boundary.LOWER, i));
                    xValues.add(((Histogram)function).getBinCenter(0, i));
                    xValues.add(((Histogram)function).getBinLimits(0, Histogram.Boundary.UPPER, i));
                    continue;
                }
                xValues.add(function.get(0, i));
            }
        }
        Collections.sort(xValues);
        return xValues;
    }

    @SafeVarargs
    public static DataSet mathFunction(DataSet function, double value, MathOp op, Formatter<Number> ... format) {
        String functionName = DataSetMath.getFormatter(format).format("{0}({1})", new Object[]{op.getTag(), function.getName()});
        double[] y = function.getValues(1);
        double[] eyn = Arrays.copyOf(DataSetMath.errors(function, ErrType.EYN), y.length);
        double[] eyp = Arrays.copyOf(DataSetMath.errors(function, ErrType.EYP), y.length);
        int ncount = function.getDataCount();
        switch (op) {
            case ADD: {
                return new DoubleErrorDataSet(functionName, function.getValues(0), ArrayMath.add(y, value), eyn, eyp, ncount, true);
            }
            case SUBTRACT: {
                return new DoubleErrorDataSet(functionName, function.getValues(0), ArrayMath.subtract(y, value), eyn, eyp, ncount, true);
            }
            case MULTIPLY: {
                return new DoubleErrorDataSet(functionName, function.getValues(0), ArrayMath.multiply(y, value), ArrayMath.multiply(eyn, value), ArrayMath.multiply(eyp, value), ncount, true);
            }
            case DIVIDE: {
                return new DoubleErrorDataSet(functionName, function.getValues(0), ArrayMath.divide(y, value), ArrayMath.divide(eyn, value), ArrayMath.divide(eyp, value), ncount, true);
            }
            case SQR: {
                for (int i = 0; i < eyn.length; ++i) {
                    eyn[i] = 2.0 * MathBase.abs(y[i] + value) * eyn[i];
                    eyp[i] = 2.0 * MathBase.abs(y[i] + value) * eyp[i];
                }
                return new DoubleErrorDataSet(functionName, function.getValues(0), value == 0.0 ? ArrayMath.sqr(y) : ArrayMath.sqr(ArrayMath.add(y, value)), eyn, eyp, ncount, true);
            }
            case SQRT: {
                for (int i = 0; i < eyn.length; ++i) {
                    eyn[i] = MathBase.sqrt(MathBase.abs(y[i] + value)) * eyn[i];
                    eyp[i] = MathBase.sqrt(MathBase.abs(y[i] + value)) * eyp[i];
                }
                return new DoubleErrorDataSet(functionName, function.getValues(0), value == 0.0 ? ArrayMath.sqrt(y) : ArrayMath.sqrt(ArrayMath.add(y, value)), eyn, eyp, ncount, true);
            }
            case LOG10: {
                for (int i = 0; i < eyn.length; ++i) {
                    eyn[i] = 0.0;
                    eyp[i] = 0.0;
                }
                return new DoubleErrorDataSet(functionName, function.getValues(0), ArrayMath.tenLog10(y), eyn, eyp, ncount, true);
            }
            case DB: {
                for (int i = 0; i < eyn.length; ++i) {
                    eyn[i] = 0.0;
                    eyp[i] = 0.0;
                }
                return new DoubleErrorDataSet(functionName, function.getValues(0), ArrayMath.decibel(y), eyn, eyp, ncount, true);
            }
            case INV_DB: {
                for (int i = 0; i < eyn.length; ++i) {
                    eyn[i] = 0.0;
                    eyp[i] = 0.0;
                }
                return new DoubleErrorDataSet(functionName, function.getValues(0), ArrayMath.inverseDecibel(y), eyn, eyp, ncount, true);
            }
        }
        return new DoubleErrorDataSet(functionName, function.getValues(0), function.getValues(1), DataSetMath.errors(function, ErrType.EYN), DataSetMath.errors(function, ErrType.EYP), ncount, true);
    }

    @SafeVarargs
    public static DataSet maxFilteredFunction(DataSet function, double width, Formatter<Number> ... format) {
        return DataSetMath.filterFunction(function, width, Filter.MAX, format);
    }

    @SafeVarargs
    public static DataSet medianFilteredFunction(DataSet function, double width, Formatter<Number> ... format) {
        return DataSetMath.filterFunction(function, width, Filter.MEDIAN, format);
    }

    @SafeVarargs
    public static DataSet minFilteredFunction(DataSet function, double width, Formatter<Number> ... format) {
        return DataSetMath.filterFunction(function, width, Filter.MIN, format);
    }

    @SafeVarargs
    public static DataSet multiplyFunction(DataSet function1, DataSet function2, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function1, function2, MathOp.MULTIPLY, format);
    }

    @SafeVarargs
    public static DataSet multiplyFunction(DataSet function, double value, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, value, MathOp.MULTIPLY, format);
    }

    @SafeVarargs
    public static DataSet normalisedFunction(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.normalisedFunction(function, 1.0, format);
    }

    @SafeVarargs
    public static DataSet normalisedFunction(DataSet function, double requiredIntegral, Formatter<Number> ... format) {
        DoublePointError complexInt = DataSetMath.integral(function);
        double integral = complexInt.getY() / requiredIntegral;
        int ncount = function.getDataCount();
        String functionName = DataSetMath.getFormatter(format).format("{0}", new Object[]{function.getName()});
        if (integral == 0.0) {
            return new DoubleErrorDataSet(functionName, function.getValues(0), new double[ncount], new double[ncount], new double[ncount], ncount, true);
        }
        double[] xValues = function.getValues(0);
        double[] yValues = ArrayMath.divide(function.getValues(1), integral);
        double[] eyp = ArrayMath.divide(DataSetMath.errors(function, ErrType.EYN), integral);
        double[] eyn = ArrayMath.divide(DataSetMath.errors(function, ErrType.EYP), integral);
        return new DoubleErrorDataSet(functionName, xValues, yValues, eyp, eyn, ncount, true);
    }

    @SafeVarargs
    public static DataSet normalisedMagnitudeSpectrumDecibel(DataSet function, Formatter<Number> ... format) {
        return DataSetMath.magnitudeSpectrum(function, Apodization.Hann, true, true, format);
    }

    @SafeVarargs
    public static DataSet peakToPeakFilteredFunction(DataSet function, double width, Formatter<Number> ... format) {
        return DataSetMath.filterFunction(function, width, Filter.P2P, format);
    }

    @SafeVarargs
    public static DataSet rmsFilteredFunction(DataSet function, double width, Formatter<Number> ... format) {
        return DataSetMath.filterFunction(function, width, Filter.RMS, format);
    }

    public static EditableDataSet setFunction(EditableDataSet function, double value, double xMin, double xMax) {
        int nLength = function.getDataCount();
        double xMinLocal = function.get(0, 0);
        double xMaxLocal = function.get(0, nLength - 1);
        if (Double.isFinite(xMin) && Double.isFinite(xMax)) {
            xMinLocal = MathBase.min(xMin, xMax);
            xMaxLocal = MathBase.max(xMin, xMax);
        } else if (Double.isFinite(xMin)) {
            xMinLocal = xMin;
        } else if (Double.isFinite(xMax)) {
            xMaxLocal = xMax;
        }
        boolean oldState = function.autoNotification().getAndSet(false);
        for (int i = 0; i < nLength; ++i) {
            double x = function.get(0, i);
            if (!(x >= xMinLocal) || !(x <= xMaxLocal)) continue;
            function.set(i, new double[]{x, value});
        }
        function.autoNotification().set(oldState);
        return function;
    }

    @SafeVarargs
    public static DataSet sqrFunction(DataSet function, double value, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, value, MathOp.SQR, format);
    }

    @SafeVarargs
    public static DataSet sqrFunction(DataSet function1, DataSet function2, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function1, function2, MathOp.SQR, format);
    }

    @SafeVarargs
    public static DataSet sqrtFunction(DataSet function, double value, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, value, MathOp.SQRT, format);
    }

    @SafeVarargs
    public static DataSet sqrtFunction(DataSet function1, DataSet function2, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function1, function2, MathOp.SQRT, format);
    }

    @SafeVarargs
    public static DataSet subtractFunction(DataSet function1, DataSet function2, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function1, function2, MathOp.SUBTRACT, format);
    }

    @SafeVarargs
    public static DataSet subtractFunction(DataSet function, double value, Formatter<Number> ... format) {
        return DataSetMath.mathFunction(function, value, MathOp.SUBTRACT, format);
    }

    @SafeVarargs
    private static Formatter<Number> getFormatter(Formatter<Number> ... format) {
        return Objects.requireNonNull(format, "user-supplied format").length > 0 ? format[0] : DEFAULT_FORMATTER;
    }

    private static double[] cropToLength(double[] in, int length) {
        if (in.length == length) {
            return in;
        }
        return Arrays.copyOf(in, length);
    }

    public static enum MathOp {
        ADD("+"),
        SUBTRACT("-"),
        MULTIPLY("*"),
        DIVIDE("*"),
        SQR("SQR"),
        SQRT("SQRT"),
        LOG10("Log10"),
        DB("dB"),
        INV_DB("dB^{-1}"),
        IDENTITY("Copy");

        private final String tag;

        private MathOp(String tag) {
            this.tag = tag;
        }

        public String getTag() {
            return this.tag;
        }
    }

    public static enum Filter {
        MEAN("LowPass"),
        MEDIAN("Median"),
        MIN("Min"),
        MAX("Max"),
        P2P("PeakToPeak"),
        RMS("RMS"),
        GEOMMEAN("GeometricMean");

        private final String tag;

        private Filter(String tag) {
            this.tag = tag;
        }

        public String getTag() {
            return this.tag;
        }
    }

    public static enum ErrType {
        EXN,
        EXP,
        EYN,
        EYP;

    }
}

